//----------------------------------------------------------------------------
//   The confidential and proprietary information contained in this file may
//   only be used by a person authorised under and to the extent permitted
//   by a subsisting licensing agreement from ARM Limited or its affiliates.
//
//          (C) COPYRIGHT 2017 ARM Limited or its affiliates.
//              ALL RIGHTS RESERVED
//
//   This entire notice must be reproduced on all copies of this file
//   and copies of this file may only be made by a person if such person is
//   permitted to do so under the terms of a subsisting license agreement
//   from ARM Limited or its affiliates.
//----------------------------------------------------------------------------
'use strict';
const express = require('express');
const fs = require('fs');
const router = express.Router();
const events = require('events'); // jshint ignore:line
const runScrTemplate = require('./template');
const path = require('path');
const pkg = require('./package.json');

var app = express();
app.set('verboseLog', process.env.LIBGDC_LOG || false);


router.get('/', function(req, res) {
    let html = fs.readFileSync(__dirname + '/static/html/index.html','utf8');
    html = html.replace('@@VERSION', pkg.version);
    res.send(html);
});


function toTypedArray(buffer) {
    let ab = new ArrayBuffer(buffer.length);
    let view = new Uint8Array(ab);
    let i = 0;
    while (i < buffer.length) {
        view[i] = buffer[i];
        i++;
    }
    return view;
}


function toBuffer(ab) {
    let buffer = new Buffer(ab.byteLength);
    let view = new Uint8Array(ab);
    let i = 0;
    while (i < buffer.length) {
        buffer[i] = view[i];
        i++;
    }
    return buffer;
}

function checkForCustomData(layout, index, callback) {
    if (layout.transformations[index]) {
        if (layout.transformations[index].transformation === 'Custom') {
            let filename = 'config' + index + '.txt';
            fs.writeFile('./libraries/libapical-gdc/' + filename, layout.transformations[index].data, function(err) {
                delete layout.transformations[index].data;
                layout.transformations[index].param = {};
                layout.transformations[index].param.customTransformation = '../libapical-gdc/' + filename;
                index++;
                checkForCustomData(layout, index, callback);
            });
        } else {
            index++;
            checkForCustomData(layout, index, callback);
        }
    } else {
        callback(layout);
    }
}

function writeFile(path, buffer) {
    return new Promise((resolve, reject) => {
        fs.writeFile(path, buffer, (err) => {
            if (err) {
                reject(err);
            } else {
                resolve();
            }
        });
    })
}

function readFile(path, encoding) {
    return new Promise((resolve, reject) => {
        fs.readFile(path, encoding, (err, data) => {
            if (err) {
                reject(err);
            } else {
                resolve(data);
            }
        });
    });
}

function executeCommand(command, options) {
    let exec = require('child_process').exec;
    return new Promise((resolve, reject) => {
        exec(command, options, function(err, stdout) {
            if (app.get('verboseLog')) {
                console.log(stdout);
            }
            if (err) {
                reject(err);
            } else {
                resolve(stdout);
            }
        })
    })
}

function clamp(n) {
    if (n < 0) {
        return 0;
    }
    if (n > 255) {
        return 255;
    }
    return Math.floor(n);
}

function alignTo16Bytes(arr, width, height) {
    if ((arr.length / height) % 16) {
        let lineOffset = Math.ceil((arr.length / height) / 16) * 16;
        let outputArr = new Uint8Array(lineOffset * height);
        let bytePadding = lineOffset - width;
        let counter = 0;
        for (let i = 0; i < arr.length; i++) {
            outputArr[counter] = arr[i];
            counter++;
            if (i && !(i % width)) {
                counter += bytePadding;
            }
        }
        return outputArr;
    }
    return arr;
}

function unalignFrom16Bytes(arr, width, height) {
    if (width % 16) {
        let padding = (Math.ceil(width / 16) * 16) - width;
        let newArr = new Uint8Array(width * height);
        let byteCounter = 0;
        for (let i = 0; i < arr.length; i++) {
            newArr[byteCounter] = arr[i];
            byteCounter++;
            if (!(byteCounter % width)) {
                i += padding;
            }
        }
        return newArr;
    }
    return arr;
}


function convertFromRGBA(mode, colourspace, width, height, data) {
    if (mode === 'planar420' || mode === 'semiplanar420' || mode === 'luminance') {
        let arr = new Uint8Array(data);
        let R;
        let G;
        let B;
        let Y = new Uint8Array(arr.length / 4);
        let U = new Uint8Array(arr.length / 4);
        let V = new Uint8Array(arr.length / 4);
        let i = 0;
        let counter = 0;
        while (i < arr.length) {
            R = arr[i];
            G = arr[i + 1];
            B = arr[i + 2];
            Y[counter] = clamp((R * 0.257) + (G * 0.504) + (B * 0.098) + 16);
            U[counter] = clamp(-(R * 0.148) - (G * 0.291) + (B * 0.439) + 128);
            V[counter] = clamp((R * 0.439) - (G * 0.368) - (B * 0.071) + 128);
            counter++;
            i += 4;
        }
        counter = 0;
        let newUData = new Uint8Array(U.length / 4);
        let newVData = new Uint8Array(V.length / 4);
        for (i = 0; i < height; i += 2) {
            for (let j = 0; j < width; j += 2) {
                newUData[counter] = ((U[j + i * width] + U[j + (i + 1) * width]) / 2);
                newVData[counter] = ((V[j + i * width] + V[j + (i + 1) * width]) / 2);
                counter++;
            }
        }
        if (mode === 'planar420') {
            return [
                ['Y.raw', alignTo16Bytes(Y, width, height)],
                ['U.raw', alignTo16Bytes(newUData, width / 2, height / 2)],
                ['V.raw', alignTo16Bytes(newVData, width / 2, height / 2)]
            ];
        } else if (mode === 'semiplanar420') {
            let uvDataBuffer = new ArrayBuffer(newUData.length + newVData.length);
            let uvData = new Uint8Array(uvDataBuffer);
            for (let i = 0; i < newUData.length; i++) {
                uvData[i * 2] = newUData[i];
                uvData[(i * 2) + 1] = newVData[i];
            }
            return [
                ['Y.raw', alignTo16Bytes(Y, width, height)],
                ['UV.raw', alignTo16Bytes(uvData, width, height / 2)]
            ];
        } else if (mode === 'luminance') {
            return [
                ['Y.raw', alignTo16Bytes(Y, width, height)]
            ]
        }
    } else if (mode === 'planar444') {
        if (colourspace === 'rgb') {
            let arr = new Uint8Array(data);
            let R = [];
            let G = [];
            let B = [];
            let i = 0;
            while (i < arr.length) {
                R.push(arr[i]);
                G.push(arr[i + 1]);
                B.push(arr[i + 2]);
                i += 4;
            }
            return [
                ['R.raw', alignTo16Bytes(Uint8Array.from(R), width, height)],
                ['G.raw', alignTo16Bytes(Uint8Array.from(G), width, height)],
                ['B.raw', alignTo16Bytes(Uint8Array.from(B), width, height)]
            ]
        } else {
            let arr = new Uint8Array(data);
            let R;
            let G;
            let B;
            let Y = new Uint8Array(arr.length / 4);
            let U = new Uint8Array(arr.length / 4);
            let V = new Uint8Array(arr.length / 4);
            let i = 0;
            let counter = 0;
            while (i < arr.length) {
                R = arr[i];
                G = arr[i + 1];
                B = arr[i + 2];
                Y[counter] = clamp((R * 0.257) + (G * 0.504) + (B * 0.098) + 16);
                U[counter] = clamp(-(R * 0.148) - (G * 0.291) + (B * 0.439) + 128);
                V[counter] = clamp((R * 0.439) - (G * 0.368) - (B * 0.071) + 128);
                counter++;
                i += 4;
            }
            return [
                ['Y.raw', alignTo16Bytes(Uint8Array.from(Y), width, height)],
                ['U.raw', alignTo16Bytes(Uint8Array.from(U), width, height)],
                ['V.raw', alignTo16Bytes(Uint8Array.from(V), width, height)]
            ]
        }
    }
}

function convertToRGBA(mode, colourspace, width, height, dataArrays) {
    if (mode === 'luminance') {
        let yArray = unalignFrom16Bytes(toTypedArray(dataArrays[0]), width, height);
        let RGBA = [];
        let i = 0;
        while (i < yArray.length) {
            RGBA.push(
                yArray[i],
                yArray[i],
                yArray[i],
                255
            );
            i++;
        }
        return toBuffer(Uint8Array.from(RGBA).buffer);
    }
    if (mode === 'planar420') {
        let yArray = unalignFrom16Bytes(toTypedArray(dataArrays[0]), width, height);
        let uArray = unalignFrom16Bytes(toTypedArray(dataArrays[1]), width / 2, height / 2);
        let vArray = unalignFrom16Bytes(toTypedArray(dataArrays[2]), width / 2, height / 2);
        let newUMatrix = [];
        let newVMatrix = [];
        let uTemp, vTemp;
        let uVal, vVal;
        for (let i = 0; i < height / 2; i++) {
            uTemp = [];
            vTemp = [];
            for (let j = 0; j < (width / 2); j++) {
                uVal = uArray[j + (i * (width / 2))];
                vVal = vArray[j + (i * (width / 2))];
                uTemp.push(uVal, uVal);
                vTemp.push(vVal, vVal);
            }
            newUMatrix.push(uTemp, uTemp);
            newVMatrix.push(vTemp, vTemp);
        }

        let newUArray = Uint8Array.from([].concat.apply([], newUMatrix));
        let newVArray = Uint8Array.from([].concat.apply([], newVMatrix));

        let RGBA = [];
        let i = 0;
        while (i < yArray.length) {
            RGBA.push(
                clamp((1.164 * (yArray[i] - 16)) + (1.596 * (newVArray[i] - 128))),
                clamp((1.164 * (yArray[i] - 16)) - (0.392 * (newUArray[i] - 128)) - (0.813 * (newVArray[i] - 128))),
                clamp((1.164 * (yArray[i] - 16)) + (2.017 * (newUArray[i] - 128))),
                255
            );
            i++;
        }
        return toBuffer(Uint8Array.from(RGBA).buffer);
    } else if (mode === 'semiplanar420') {
        let yArray = unalignFrom16Bytes(toTypedArray(dataArrays[0]), width, height);
        let uvArray = unalignFrom16Bytes(toTypedArray(dataArrays[1]), width, height / 2);
        let i;
        let uArray = new Uint8Array(new Buffer(dataArrays[1].length/2));
        let vArray = new Uint8Array(new Buffer(dataArrays[1].length/2));
        for (i = 0; i < uvArray.length; i++) {
            uArray[i] = uvArray[i * 2];
            vArray[i] = uvArray[(i * 2) + 1];
        }
        let newUMatrix = [];
        let newVMatrix = [];
        let uTemp, vTemp;
        let uVal, vVal;
        for (i = 0; i < (height / 2); i++) {
            uTemp = [];
            vTemp = [];
            for (let j = 0; j < (width / 2); j++) {
                uVal = uArray[j + (i * (width / 2))];
                vVal = vArray[j + (i * (width / 2))];
                uTemp.push(uVal, uVal);
                vTemp.push(vVal, vVal);
            }
            newUMatrix.push(uTemp, uTemp);
            newVMatrix.push(vTemp, vTemp);
        }

        let newUArray = Uint8Array.from([].concat.apply([], newUMatrix));
        let newVArray = Uint8Array.from([].concat.apply([], newVMatrix));

        let RGBA = [];
        i = 0;
        while (i < yArray.length) {
            RGBA.push(
                clamp((1.164 * (yArray[i] - 16)) + (1.596 * (newVArray[i] - 128))),
                clamp((1.164 * (yArray[i] - 16)) - (0.392 * (newUArray[i] - 128)) - (0.813 * (newVArray[i] - 128))),
                clamp((1.164 * (yArray[i] - 16)) + (2.017 * (newUArray[i] - 128))),
                255
            );
            i++;
        }
        return toBuffer(Uint8Array.from(RGBA).buffer);
    } else if (mode === 'planar444') {
        if (colourspace === 'rgb') {
            let R = unalignFrom16Bytes(dataArrays[0], width, height);
            let G = unalignFrom16Bytes(dataArrays[1], width, height);
            let B = unalignFrom16Bytes(dataArrays[2], width, height);

            let i = 0;
            let RGBA = [];
            while (i < R.length) {
                RGBA.push(
                    clamp(R[i]),
                    clamp(G[i]),
                    clamp(B[i]),
                    255
                );
                i++;
            }
            return toBuffer(Uint8Array.from(RGBA).buffer);
        } else {
            let Y = unalignFrom16Bytes(dataArrays[0], width, height);
            let U = unalignFrom16Bytes(dataArrays[1], width, height);
            let V = unalignFrom16Bytes(dataArrays[2], width, height);
            let i = 0;
            let RGBA = [];
            while (i < Y.length) {
                RGBA.push(
                    clamp((1.164 * (Y[i] - 16)) + (1.596 * (V[i] - 128))),
                    clamp((1.164 * (Y[i] - 16)) - (0.392 * (U[i] - 128)) - (0.813 * (V[i] - 128))),
                    clamp((1.164 * (Y[i] - 16)) + (2.017 * (U[i] - 128))),
                    255
                );
                i++;
            }
            return toBuffer(Uint8Array.from(RGBA).buffer);
        }
    }
}


function getIOFileConfig(mode, colourSpace, inputWidth, outputWidth, outputHeight) {
    if (mode === 'planar420') {
        return {
            input: [
                {
                    name: 'Y.raw',
                    width: Math.ceil(inputWidth / 16) * 16
                },
                {
                    name: 'U.raw',
                    width: Math.ceil((inputWidth / 2) / 16) * 16
                },
                {
                    name: 'V.raw',
                    width: Math.ceil((inputWidth / 2) / 16) * 16
                }
            ],
            output: [
                {
                    name: 'out_Y.raw',
                    width: Math.ceil(outputWidth / 16) * 16,
                    size: (Math.ceil(outputWidth / 16) * 16) * outputHeight
                },
                {
                    name: 'out_U.raw',
                    width: Math.ceil((outputWidth / 2) / 16) * 16,
                    size: (Math.ceil((outputWidth / 2) / 16) * 16) * (outputHeight / 2)
                },
                {
                    name: 'out_V.raw',
                    width: Math.ceil((outputWidth / 2) / 16) * 16,
                    size: (Math.ceil((outputWidth / 2) / 16) * 16) * (outputHeight / 2)
                }
            ]
        }
    }
    if (mode === 'semiplanar420') {
        return {
            input: [
                {
                    name: 'Y.raw',
                    width: Math.ceil(inputWidth / 16) * 16
                },
                {
                    name: 'UV.raw',
                    width: Math.ceil(inputWidth / 16) * 16
                }
            ],
            output: [
                {
                    name: 'out_Y.raw',
                    width: Math.ceil(outputWidth / 16) * 16,
                    size: (Math.ceil(outputWidth / 16) * 16) * outputHeight
                },
                {
                    name: 'out_UV.raw',
                    width: Math.ceil(outputWidth / 16) * 16,
                    size: (Math.ceil(outputWidth / 16) * 16) * (outputHeight / 2)
                }
            ]
        }
    }
    if (mode === 'planar444') {
        if (colourSpace === 'yuv') {
            return {
                input: [
                    {
                        name: 'Y.raw',
                        width: Math.ceil(inputWidth / 16) * 16
                    },
                    {
                        name: 'U.raw',
                        width: Math.ceil(inputWidth / 16) * 16
                    },
                    {
                        name: 'V.raw',
                        width: Math.ceil(inputWidth / 16) * 16
                    }
                ],
                output: [
                    {
                        name: 'out_Y.raw',
                        width: Math.ceil(outputWidth / 16) * 16,
                        size: Math.ceil(outputWidth / 16) * 16 * outputHeight
                    },
                    {
                        name: 'out_U.raw',
                        width: Math.ceil(outputWidth / 16) * 16,
                        size: Math.ceil(outputWidth / 16) * 16 * outputHeight
                    },
                    {
                        name: 'out_V.raw',
                        width: Math.ceil(outputWidth / 16) * 16,
                        size: Math.ceil(outputWidth / 16) * 16 * outputHeight
                    }
                ]
            }
        }
        if (colourSpace === 'rgb') {
            return {
                input: [
                    {
                        name: 'R.raw',
                        width: Math.ceil(inputWidth / 16) * 16
                    },
                    {
                        name: 'G.raw',
                        width: Math.ceil(inputWidth / 16) * 16
                    },
                    {
                        name: 'B.raw',
                        width: Math.ceil(inputWidth / 16) * 16
                    }
                ],
                output: [
                    {
                        name: 'out_R.raw',
                        width: Math.ceil(outputWidth / 16) * 16,
                        size: Math.ceil(outputWidth / 16) * 16 * outputHeight
                    },
                    {
                        name: 'out_G.raw',
                        width: Math.ceil(outputWidth / 16) * 16,
                        size: Math.ceil(outputWidth / 16) * 16 * outputHeight
                    },
                    {
                        name: 'out_B.raw',
                        width: Math.ceil(outputWidth / 16) * 16,
                        size: Math.ceil(outputWidth / 16) * 16 * outputHeight
                    }
                ]
            }
        }
    }
    if (mode === 'luminance') {
        return {
            input: [
                {
                    name: 'Y.raw',
                    width: Math.ceil(inputWidth / 16) * 16
                }
            ],
            output: [
                {
                    name: 'out_Y.raw',
                    width: Math.ceil(outputWidth / 16) * 16,
                    size: Math.ceil(outputWidth / 16) * 16 * outputHeight
                }
            ]
        }
    }
}


router.post('/transform-image', function(req, res) {
    let body = [];
    let totalLength = 0;
    req.on('data', function (chunk) {
        totalLength += chunk.length;
        body.push(chunk);
    });
    req.on('end', function() {
        readFile('./libraries/libgdc/scene.json', 'utf8')
            .then(settingsString => {
                let settings = JSON.parse(settingsString);
                let inputWidth = settings['inputRes'][0];
                let inputHeight = settings['inputRes'][1];
                let outputWidth = settings['outputRes'][0];
                let outputHeight = settings['outputRes'][1];

                let imageByteData = Buffer.concat(body);

                let lineOffset = Math.ceil(((imageByteData.length / 4) / inputHeight) / 16) * 16;
                let bytePadding = Math.max(lineOffset - inputWidth, 0);

                let convertedData = convertFromRGBA(settings['mode'], settings['colourspace'], inputWidth, inputHeight, imageByteData, bytePadding);

                let fileWritePromises = [];

                convertedData.forEach(data => {
                    fileWritePromises.push(writeFile(`./libraries/libapical-gdc/${data[0]}`, toBuffer(data[1].buffer)));
                });



                Promise.all(fileWritePromises)
                    .then(() => {
                        let stats = fs.statSync('./libraries/libapical-gdc/config.bin');
                        let configSize = stats["size"]/4;

                        let ioFileConfig = getIOFileConfig(settings.mode, settings.colourspace, inputWidth, outputWidth, outputHeight);
                        let runscr = runScrTemplate.generate(configSize, inputWidth, inputHeight, ioFileConfig.input, outputWidth, outputHeight, ioFileConfig.output);
                        writeFile('./libraries/libapical-gdc/run.scr', runscr)
                            .then(() => {
                                let gdcPath = path.join(__dirname, '/libraries/libapical-gdc/gdc');
                                let inputPath = path.join( __dirname, '/libraries/libapical-gdc/run.scr');
                                let outputPath = path.join(__dirname, '/libraries/libapical-gdc/output.txt');
                                let cwdPath = path.join(__dirname, '/libraries/libapical-gdc');
                                executeCommand(gdcPath + ' -i ' + inputPath + ' -o ' + outputPath, {cwd: cwdPath})
                                    .then(() => {
                                        let fileReadPromises = [];
                                        convertedData.forEach(data => {
                                            fileReadPromises.push(readFile(`./libraries/libapical-gdc/out_${data[0]}`));
                                        });
                                        Promise.all(fileReadPromises)
                                            .then(dataArrays => {
                                                let buff = convertToRGBA(settings['mode'], settings['colourspace'], outputWidth, outputHeight, dataArrays, bytePadding);
                                                res.status(200);
                                                res.setHeader('Content-Type', 'application/octet-stream');
                                                res.end(buff);
                                            }).catch(err => {console.error(err)});
                                    }).catch(err => {console.error(err)});
                        }).catch(err => {console.error(err)});
                    }).catch(err => {console.error(err)});
                }).catch(err => {console.error(err)});
    });
});


router.post('/generate-grid', function(req, res) {
    var data = '';
    req.on('data', function(chunk){
        data += chunk;
    });
    req.on('end', function(){
        var settings = JSON.parse(data);
        checkForCustomData(settings, 0, function(parsedSettings) {
            fs.writeFile("./libraries/libgdc/scene.json", JSON.stringify(parsedSettings, null, 4), function(err) {
                if (err) {
                    throw err;
                }
                var exec = require('child_process').exec;
                var url = __dirname + '/libraries/libgdc/gdc ' + __dirname + '/libraries/libgdc/scene.json ' + __dirname + '/libraries/libapical-gdc/config.bin';
                if (settings.mode === 'semi-planar') {
                    url += ' -s';
                }
                switch (settings.eccMode) {
                    case 'eccOnly':
                        url += ' -ecc 1';
                        break;
                    case 'eccEmbedded':
                        url += ' -ecc 2';
                        break;
                    default:
                        url += ' -ecc 0';
                }
                url += ' -g';
                exec(url, {cwd: __dirname + '/libraries/libgdc'}, function(err, stdout) {
                    if (app.get('verboseLog')) {
                        console.log(stdout);
                    }
                    if (err) {
                        console.log('native process error: ' + err);
                        res.status(500);
                        res.send('native process error code: ' + err.code);
                    } else {
                        res.status(200);
                        res.send(stdout);
                    }
                });
            });
        });
    });
});

router.post('/generate-binary', function (req, res) {
    var data = '';
    req.on('data', function (chunk) {
        data += chunk;
    });
    req.on('end', function () {
        var settings = JSON.parse(data);
        checkForCustomData(settings, 0, function (parsedSettings) {
            fs.writeFile("./libraries/libgdc/scene.json", JSON.stringify(parsedSettings, null, 4), function (err) {
                if (err) {
                    throw err;
                }
                var exec = require('child_process').exec;
                var url = __dirname + '/libraries/libgdc/gdc ' + __dirname + '/libraries/libgdc/scene.json ' + __dirname + '/libraries/libapical-gdc/config.bin';
                if (settings.mode === 'semi-planar') {
                    url += ' -s';
                }
                switch (settings.eccMode) {
                    case 'eccOnly':
                        url += ' -ecc 1';
                        break;
                    case 'eccEmbedded':
                        url += ' -ecc 2';
                        break;
                    default:
                        url += ' -ecc 0';
                }
                try {
                    exec(url, {
                        cwd: __dirname + '/libraries/libgdc'
                    }, function (err, stdout) {
                        if (err) {
                            console.log('native process error: ' + err);
                            res.status(500);
                            res.send('native process error code: ' + err.code);
                        } else {
                            var path = __dirname + '/libraries/libapical-gdc/config.bin';
                            res.setHeader('Content-Type', 'application/octet-stream');
                            res.setHeader('responseType', 'arraybuffer');
                            var f = fs.readFileSync(path);
                            res.send(f);
                        }
                    });
                } catch (err) {
                    console.log('native process error: ' + err);
                    res.status(500);
                    res.send('native process error code: ' + err.code);
                }
            });

        });
    });
});


router.post('/upload-custom-settings', function(req, res) {
    var newPath = __dirname + "/customSettings.json";
    var writeStream = fs.createWriteStream(newPath);
    req.pipe(writeStream);
    res.send('success');
});


module.exports = router;
