//----------------------------------------------------------------------------
//   The confidential and proprietary information contained in this file may
//   only be used by a person authorised under and to the extent permitted
//   by a subsisting licensing agreement from ARM Limited or its affiliates.
//
//          (C) COPYRIGHT 2017 ARM Limited or its affiliates.
//              ALL RIGHTS RESERVED
//
//   This entire notice must be reproduced on all copies of this file
//   and copies of this file may only be made by a person if such person is
//   permitted to do so under the terms of a subsisting license agreement
//   from ARM Limited or its affiliates.
//----------------------------------------------------------------------------
function Defer() {
	if (isPromise()) {
		return Promise.defer();
	} else {
		this.resolve = null;
		this.reject = null;

		this.promise = createPromise(this);
		Object.freeze(this);
	}

	function createPromise(p) {
		return new Promise(function (resolve, reject) {
			p.resolve = resolve;
			p.reject = reject;
		}.bind(p));
	}

	function isPromise() {
		return typeof (Promise) !== 'undefined' && Promise.defer;
	}
}

var actions = {
    deferAction: function(name, data) {
        var deferred = new Defer();
        document.dispatchEvent(new CustomEvent(name, {detail: {data: data, deferred: deferred}}));
        return deferred.promise;
    },
    handleAction: function(name, func) {
        document.addEventListener(name, function(event) {
            func(event.detail.data, event.detail.deferred);
        });
    }
};

(function init() {

    if (!window.indexedDB) {
        throw 'ERROR: This browser does not support indexedDB!';
    }

    var CURRENT_DB = null;

    // Creating/Removing Databases

    // Remove entire database with the given name.
    function removeDatabase(name, deferred) {
        var request = window.indexedDB.deleteDatabase(name);
        request.onsuccess = function() {
            deferred.resolve();
        };
        request.onerror = deferred.reject;
    }

    actions.handleAction('storage:remove-database', function(data, deferred) {
        removeDatabase(data.databaseName, deferred);
    });


    // Create a database with the given name and structure (optional)
    function createDatabase(name, structure, deferred) {
        var request = window.indexedDB.open(name);
        request.onupgradeneeded = function(event) {
            var db = event.target.result;
            if (structure) {
                for (var objectStoreName in structure) {
                    if (structure.hasOwnProperty(objectStoreName)) {
                        var objectStore = db.createObjectStore(objectStoreName, structure[objectStoreName].attr);
                        for (var key in structure[objectStoreName].defaults) {
                            if (structure[objectStoreName].defaults.hasOwnProperty(key)) {
                                objectStore.add(structure[objectStoreName].defaults[key], key);
                            }
                        }
                    }
                }
            }
        };
        request.onsuccess = function() {
            CURRENT_DB = name;
            deferred.resolve();
        };
        request.onerror = deferred.reject;
    }

    actions.handleAction('storage:create-database', function(data, deferred) {
        createDatabase(data.databaseName, data.structure, deferred);
    });

    // Accessing the database - functions providing access to the database, it's object stores and cursor creation.

    // Opens the database with the given name.
    function openDatabase(dabaseName) {
        var deferred = new Defer();
        var request = indexedDB.open(dabaseName);
        request.onsuccess = function(event) {
            deferred.resolve(event.target.result);
        };
        request.onerror = deferred.reject;
        return deferred.promise;
    }

    // Storing Data Objects

    // Stores an object in an object store at a specific key.
    function addObject(objectStoreName, object, key, deferred) {
        openDatabase(CURRENT_DB).then(function(db) {
            var store = db.transaction(objectStoreName, 'readwrite').objectStore(objectStoreName);
            var request;
            if (key !== null && key !== undefined && key !== '') {
                request = store.add(object, key);
            } else {
                request = store.add(object);
            }
            request.onsuccess = deferred.resolve;
            request.onerror = deferred.reject;
        });
    }

    actions.handleAction('storage:add-object', function(data, deferred) {
         addObject(data.objectStoreName, data.object, data.key, deferred);
    });

    // Retrieving Data Objects

    // Gets the object stored at a given key
    function getObjectByKey(objectStoreName, key, deferred) {
        openDatabase(CURRENT_DB).then(function(db) {
                var store = db.transaction(objectStoreName, 'readwrite').objectStore(objectStoreName);
                var request = store.get(key);
                request.onsuccess = function(event) {
                    deferred.resolve(event.target.result);
                };
                request.onerror = deferred.reject;
            });
    }

    actions.handleAction('storage:get-object-by-key', function(data, deferred) {
        getObjectByKey(data.objectStoreName, data.key, deferred);
    });

    // Gets all objects from an object store as a key/value javascript object.
    function getObjects(objectStoreName, deferred) {
        var data = {};
        openDatabase(CURRENT_DB).then(function(db) {
                var store = db.transaction(objectStoreName, 'readwrite').objectStore(objectStoreName);
                store.openCursor().onsuccess = function(event) {
                var cursor = event.target.result;
                if (cursor) {
                    data[cursor.key] = cursor.value;
                    cursor.continue();
                }
                else {
                    deferred.resolve(data);
                }
            };
        })
        .catch(deferred.reject);
    }

    actions.handleAction('storage:get-objects', function(data, deferred) {
        getObjects(data.objectStoreName, deferred);
    });

    // Gets all objects from an object store as an arraylist of data objects.
    function getObjectsAsList(objectStoreName, deferred) {
        var data = [];
        openDatabase(CURRENT_DB).then(function(db) {
                var store = db.transaction(objectStoreName, 'readwrite').objectStore(objectStoreName);
                store.openCursor().onsuccess = function(event) {
                var cursor = event.target.result;
                if (cursor) {
                    data.push(cursor.value);
                    cursor.continue();
                } else {
                    deferred.resolve(data);
                }
            };
        })
        .catch(deferred.reject);
    }

    actions.handleAction('storage:get-objects-as-list', function(data, deferred) {
        getObjectsAsList(data.objectStoreName, deferred);
    });

    // Updating Data Objects

    // Updates a data object stored at a key in an object store.
    function updateObjectByKey(objectStoreName, key, object, deferred) {
        openDatabase(CURRENT_DB).then(function(db) {
                var store = db.transaction(objectStoreName, 'readwrite').objectStore(objectStoreName);
                var request = store.put(object, key);
                request.onsuccess = function() {
                    deferred.resolve();
                };
                request.onerror = deferred.reject;
            });
    }

    actions.handleAction('storage:update-object-by-key', function(data, deferred) {
        updateObjectByKey(data.objectStoreName, data.key, data.object, deferred);
    });

    // Updates a data object at an index in an object store.
    function updateObjectByIndex(objectStoreName, index, object, deferred) {
        var i = 0;
        openDatabase(CURRENT_DB).then(function(db) {
                var store = db.transaction(objectStoreName, 'readwrite').objectStore(objectStoreName);
                store.openCursor().onsuccess = function(event) {
                var cursor = event.target.result;
                if (cursor) {
                    if (index === i) {
                        cursor.update(object);
                        deferred.resolve();
                    } else {
                        i++;
                        cursor.continue();
                    }
                }
                else {
                    deferred.reject();
                }
            };
        });
    }

    actions.handleAction('storage:update-object-by-index', function(data, deferred) {
        updateObjectByIndex(data.objectStoreName, data.index, data.object, deferred);
    });

    // Updates multiple objects at once by taking an object containing pairs of keys and data objects to update.
    function updateObjects(objectStoreName, objects, deferred) {
        openDatabase(CURRENT_DB).then(function(db) {
                var store = db.transaction(objectStoreName, 'readwrite').objectStore(objectStoreName);
                store.openCursor().onsuccess = function(event) {
                var cursor = event.target.result;
                if (cursor) {
                    if (cursor.key in objects) {
                        cursor.update(objects[cursor.key]);
                        delete objects[cursor.key];
                    }
                    cursor.continue();
                }
                else {
                    deferred.resolve();
                }
            };
        })
        .catch(deferred.reject);
    }

    actions.handleAction('storage:update-objects', function(data, deferred) {
        updateObjects(data.objectStoreName, data.objects, deferred);
    });

    // Removing Data Objects

    // Removes an object by it's index within it's object store.
    function removeObjectByIndex(objectStoreName, index, deferred) {
        var i = 0;
        openDatabase(CURRENT_DB).then(function(db) {
                var store = db.transaction(objectStoreName, 'readwrite').objectStore(objectStoreName);
                store.openCursor().onsuccess = function(event) {
                var cursor = event.target.result;
                if (cursor) {
                    if (index === i) {
                        cursor.delete();
                        deferred.resolve();
                    } else {
                        i++;
                        cursor.continue();
                    }
                }
                else {
                    deferred.reject();
                }
            };
        });
    }

    actions.handleAction('storage:remove-object-by-index', function(data, deferred) {
        removeObjectByIndex(data.objectStoreName, data.index, deferred);
    });

    // Removes an object by the key that it is stored under within it's object store.
    function removeObjectByKey(objectStoreName, key, deferred) {
        openDatabase(CURRENT_DB).then(function(db) {
                var store = db.transaction(objectStoreName, 'readwrite').objectStore(objectStoreName);
                store.getAllKeys().onsuccess = function() {
                var keys = event.target.result;
                var found = false;
                for (var i = 0; i < keys.length; i++) {
                    if (keys[i] === key) {
                        store.delete(key);
                        deferred.resolve();
                    }
                }
                deferred.reject();
            };
        });
    }

    actions.handleAction('storage:remove-object-by-key', function(data, deferred) {
        removeObjectByKey(data.objectStoreName, data.key, deferred);
    });

    // Clears all data from an object store.
    function clearObjectStore(objectStoreName, deferred) {
        openDatabase(CURRENT_DB).then(function(db) {
            var store = db.transaction(objectStoreName, 'readwrite').objectStore(objectStoreName);
            var request = store.clear();
            request.onsuccess = function() {
                deferred.resolve();
            };
            request.onerror = deferred.reject;
        });
    }

    actions.handleAction('storage:clear-object-store', function(data, deferred) {
        clearObjectStore(data.objectStoreName, deferred);
    });

})();

var events = {
    addEventListener: function (name, f) {
        document.addEventListener(name, function(event) {
            console.log(event);
            if (name === 'click' || name === 'change' || name === 'mouseup' || name === 'keyup') {
                f(event);
            } else {
                f(event.detail.data, event.detail.callback);
            }
        });
    },
    dispatchEvent: function (name, data, callback) {
        var eventData;
        eventData = {detail: {data: data, callback: callback}};
        document.dispatchEvent(new CustomEvent(name, eventData));
    }
};

events.addEventListener('click', function(event) {
    var target = event.target.closest('[data-click-event]');
    if (target) {
        var eventName = target.getAttribute('data-click-event');
        if (eventName) {
            events.dispatchEvent(eventName, target);
        } else {
            console.error(`no event named  ${eventName}  exists`);
        }
        event.stopPropagation();
    }
});

events.addEventListener('change', function(event) {
    var target = event.target.closest('[data-change-event]');
    if (target) {
        var eventName = target.getAttribute('data-change-event');
        if (eventName) {
            events.dispatchEvent(eventName, target);
        }
        event.stopPropagation();
    }
});

events.addEventListener('keyup', function(event) {
    if (event.keyCode === 13) {
        var target = event.target.closest('[data-blur-event]');
        if (target) {
            var eventName = target.getAttribute('data-blur-event');
            if (eventName) {
                events.dispatchEvent(eventName, target);
            }
            event.stopPropagation();
        }
    }
});

var utils = {
    getBestFitScale: function(originalWidth, originalHeight, containerWidth, containerHeight) {
        return Math.min(containerWidth/originalWidth, containerHeight/originalHeight);
    },

    snap: function(numberToSnap, step) {
        var r = numberToSnap % step;
        if (r === 0) {
            return numberToSnap;
        }
        return numberToSnap - r;
    },

    scaleLine: function(lineData, factor) {
        var scaledLine = [];
        lineData.map(function(coords) {
            scaledLine.push([
                coords[0] * factor,
                coords[1] * factor
            ]);
        });
        return scaledLine;
    },

    offsetLine: function(lineData, offsetX, offsetY) {
        var offsetLine = [];
        lineData.map(function(coords) {
            offsetLine.push([
                coords[0] + offsetX,
                coords[1] + offsetY
            ]);
        });
        return offsetLine;
    },

    throttle: function(fn, threshhold, scope) {
        threshhold = threshhold || (threshhold = 250);
        var last,
        deferTimer;
        return function () {
            var context = scope || this;
            var now = +new Date(),
            args = arguments;
            if (last && now < last + threshhold) {
                clearTimeout(deferTimer);
                deferTimer = setTimeout(function () {
                    last = now;
                    fn.apply(context, args);
                }, threshhold);
            } else {
                last = now;
                fn.apply(context, args);
            }
        };
    }
};

function Draw() {
    var that = this;
    var svgNS = 'http://www.w3.org/2000/svg';
    var TYPE = {
        SVG: 'svg',
        DEFS: 'defs',
        GROUP: 'g',
        RECT: 'rect',
        CIRCLE: 'circle',
        POLYLINE: 'polyline',
        POLYGON: 'polygon',
        TEXT: 'text',
        FILTER: 'filter',
        BLUR: 'feGaussianBlur',
        TITLE: 'title',
        IMAGE: 'image',
        CLIP_PATH: 'clipPath'
    };
    var drawObject = function(type, attributes) {
        var obj = document.createElementNS(svgNS, type);
        for (var attributeName in attributes) {
            if (attributes.hasOwnProperty(attributeName)) {
                obj.setAttributeNS(null, attributeName, attributes[attributeName]);
            }
        }
        return obj;
    };
    that.svg = function(attributes) {
        var svg = drawObject(TYPE.SVG, attributes);
        return svg;
    };
    that.group = function(attributes) {
        var group = drawObject(TYPE.GROUP, attributes);
        return group;
    };
    that.rect = function(attributes) {
        return drawObject(TYPE.RECT, attributes);
    };
    that.circle = function(attributes) {
        return drawObject(TYPE.CIRCLE, attributes);
    };
    that.polyline = function(points, attributes) {
        var polyline = drawObject(TYPE.POLYLINE, attributes);
        var pointsString = '';
        for (var i = 0; i < points.length; i++) {
            pointsString += points[i][0] + ',' + points[i][1] + ' ';
        }
        polyline.setAttributeNS(null, 'points', pointsString);
        return polyline;
    };
    that.polygon = function(points, attributes) {
        var polygon = drawObject(TYPE.POLYGON, attributes);
        var pointsString = '';
        for (var i = 0; i < points.length; i++) {
            pointsString += points[i][0] + ',' + points[i][1] + ' ';
        }
        polygon.setAttributeNS(null, 'points', pointsString);
        return polygon;
    };
    that.text = function(text, attributes) {
        var textObject = drawObject(TYPE.TEXT, attributes);
        var textNode = document.createTextNode(text);
        textObject.appendChild(textNode);
        return textObject;
    };
    that.blurFilter = function(id, source, deviation) {
        var filter = drawObject(TYPE.FILTER, {id: id});
        var blur = drawObject(TYPE.BLUR, {in: source, stdDeviation: deviation});
        filter.appendChild(blur);
        return filter;
    };
    that.title = function(titleText) {
        var title = drawObject(TYPE.TITLE);
        title.innerHTML = titleText;
        return title;
    };
    that.image = function(src, attributes) {
        var image = drawObject(TYPE.IMAGE, attributes);
        image.setAttributeNS('http://www.w3.org/1999/xlink','href', src);
        return image;
    };
    that.defs = function(attributes) {
        return  drawObject(TYPE.DEFS, attributes);
    };
    that.clipPath = function(attributes) {
        return  drawObject(TYPE.CLIP_PATH, attributes);
    };
}

(function() {

    function selectTab(parent, index) {
        var $container = $(parent);
        $container.find('.tab').each(function(i, tab) {
            var $tab = $(tab);
            $tab.removeClass('selected');
            if (i === index) {
                $tab.addClass('selected');
            }
        });
        $container.find('.tab-content').each(function(i, tabContent) {
            var $tabContent = $(tabContent);
            $tabContent.addClass('hidden');
            if (i === index) {
                $tabContent.removeClass('hidden');
            }
        });
    }

    events.addEventListener('tabs:select-tab', function(target) {
        var tabs = $(target.closest('.tabs'));
        selectTab(tabs.parent(), tabs.children().index(target));
    });

})();

(function() {
    function setInputImage(file) {
        var r = new FileReader();
        r.onload = function() {
            var img = new Image();
            img.onload = function () {
                var inputData = {
                    name: file.name,
                    image: r.result,
                    width: this.width,
                    height: this.height,
                    fov: 180,
                    diameter: Math.min(this.width, this.height),
                    offsetX: 0,
                    offsetY: 0
                };
                var view = $('.input-view');
                var scale = utils.getBestFitScale(inputData.width, inputData.height, view.width(), view.height());
                inputData.scale = scale;
                setInputControls(inputData);
                storeInputValues(inputData);
            };
            img.src = URL.createObjectURL(file);
        };
        r.readAsDataURL(file);
    }

    function getInputData() {
        return actions.deferAction('storage:get-objects', {objectStoreName: 'input'});
    }

    function setInputControls(inputData) {
        $('.input-name').val(inputData.name);
        $('.input-width').val(+inputData.width);
        $('.input-height').val(+inputData.height);
        $('.input-fov').val(+inputData.fov);
        $('.input-diameter').val(+inputData.diameter);
        $('.input-offset-x').val(+inputData.offsetX);
        $('.input-offset-y').val(+inputData.offsetY);
    }

    function getInputControlValues() {
        return {
            width: $('.input-width').val(),
            height: $('.input-height').val(),
            fov: $('.input-fov').val(),
            diameter: $('.input-diameter').val(),
            offsetX: $('.input-offset-x').val(),
            offsetY: $('.input-offset-y').val()
        };
    }

    function storeInputValues(inputData) {
        return actions.deferAction('storage:update-objects', {objectStoreName: 'input', objects: inputData}).then(function() {
            return actions.deferAction('views:draw-input-view');
        });
    }

    function setInputData(inputData) {
        setInputControls(inputData);
        return storeInputValues(inputData);
    }

    events.addEventListener('input:upload-input-image', function(event) {
        if (event.files && event.files[0]) {
            setInputImage(event.files[0]);
        }
    });

    events.addEventListener('input:trigger-select-input-dialog', function() {
        $('#upload-input-image').click();
    });

    events.addEventListener('input:user-input-changed', function() {
        storeInputValues(getInputControlValues());
    });

    actions.handleAction('input:get-input-data', function(data, deferred) {
        getInputData().then(deferred.resolve)
        .catch(deferred.reject);
    });

    actions.handleAction('input:set-input-data', function(data, deferred) {
        setInputData(data.inputData).then(deferred.resolve)
        .catch(deferred.reject);
    });

})();

(function() {

    function getOutputData() {
        return actions.deferAction('storage:get-objects', {objectStoreName: 'output'});
    }

    function setOutputResolution(width, height) {
        var deferred = new Defer();
        storeOutputResolution(width, height)
            .then(function() {
                setOutputResolutionInputs(width, height);
                actions.deferAction('views:draw-output-view');
                deferred.resolve();
            })
            .catch(deferred.reject);
        return deferred.promise;
    }

    function storeOutputResolution(width, height) {
        var restrictionApplied = false;
        if (width < 100) {
            width = 100;
            restrictionApplied = true;
        }
        if (height < 100) {
            height = 100;
            restrictionApplied = true;
        }
        if (restrictionApplied) {
            setOutputResolutionInputs(width, height);
        }

        return actions.deferAction('storage:update-objects', {objectStoreName: 'output', objects: {width: +width, height: +height}})
        .then(function() {
            return setOutputImage(null);
        });
    }

    function setOutputResolutionInputs(width, height) {
        $('.output-width').val(width);
        $('.output-height').val(height);
    }

    function setOutputImage(url) {
        return actions.deferAction('storage:update-object-by-key', {objectStoreName: 'output', key: 'image', object: url})
        .then(function() {
            if (url !== null) {
                return setImageValidity(true);
            }
            else {
                return setImageValidity(false);
            }
        });
    }

    function setImageValidity(validity) {
        return actions.deferAction('storage:update-object-by-key', {objectStoreName: 'output', key: 'imageIsValid', object: validity});
    }

    function downloadOutputImage() {
        getOutputData().then(function(outputData) {
            if (outputData.image !== null) {
                var a = document.createElement('a');
                a.setAttribute('download', 'output.png');
                a.setAttribute('href', outputData.image);
                document.body.appendChild(a);
                a.click();
                document.body.removeChild(a);
            } else {
                console.error('Image not found.');
            }
        });
    }

    events.addEventListener('output:user-input-changed', function() {
        storeOutputResolution($('.output-width').val(), $('.output-height').val()).then(function() {
            actions.deferAction('regions:auto-arrange-regions');
            actions.deferAction('views:draw-output-view');
        });
    });
    events.addEventListener('output:save-output-image', function(eventData, callback) {
        downloadOutputImage();
    });


    actions.handleAction('output:set-output-image', function(data, deferred) {
        setOutputImage(data.url).then(deferred.resolve)
        .catch(deferred.reject);
    });
    actions.handleAction('output:clear-output-image', function(data, deferred) {
        setOutputImage(null).then(deferred.resolve)
        .catch(deferred.reject);
    });
    actions.handleAction('output:set-image-validity', function(data, deferred) {
        setImageValidity(data.validity).then(deferred.resolve)
        .catch(deferred.reject);
    });
    actions.handleAction('output:set-output-resolution', function(data, deferred) {
        setOutputResolution(data.width, data.height).then(deferred.resolve)
        .catch(deferred.reject);
    });
    actions.handleAction('output:get-output-data', function(data, deferred) {
        getOutputData().then(function(outputData) {
            return deferred.resolve(outputData);
        });
    });

})();

(function() {

    function deleteAllGrids() {
        return actions.deferAction('storage:clear-object-store', {objectStoreName: 'grids'});
    }

    function generateGrids(redraw) {
        events.dispatchEvent('views:show-loading-screens');
        var setOutputImagePromise = actions.deferAction('output:set-output-image', {url: null});
        var deleteAllGridsPromise = deleteAllGrids();
        var getLayoutPromise = actions.deferAction('layout:get-layout');
        var getInputDataPromise = actions.deferAction('input:get-input-data');
        return setOutputImagePromise.then(function() {
            return deleteAllGridsPromise;
        }).then(function() {
            return Promise.all([getLayoutPromise, getInputDataPromise]);
        }).then(values => {
            var layoutData = values[0];
            var inputData = values[1];
            return $.ajax({
                url: '/generate-grid',
                type: 'POST',
                data: JSON.stringify(layoutData)
            }).done(function(result) {
                var lines = result.split('\n');
                var grids = [];
                var grid;
                var vals;
                var regionsIndex = 0;
                var col = 0;
                var row = [];
                var x, y;
                for (var i = 0; i < lines.length; i++) {
                    vals = lines[i].split(' ');
                    if (lines[i][0] !== 'T') {
                        if (lines[i][0] === 'W') {
                            var xStep = +(vals[3].replace(/(^,)|(,$)/g, ""));
                            var yStep = +(vals[5].replace(/(^,)|(,$)/g, ""));
                            var regionWidth =  layoutData.transformations[regionsIndex].position[2];
                            var xPoints = Math.ceil(regionWidth/xStep);
                            if (grid) {
                                grids.push(grid);
                            }
                            grid = {
                                xStep: xStep,
                                yStep: yStep,
                                matrix: [],
                                valid: true
                            };
                            regionsIndex++;
                        } else {
                            x = +vals[0];
                            y = +vals[1];
                            if (x < 0 || y < 0 || x > inputData.width || y > inputData.height) {
                                grid.valid = false;
                            } else {
                                row.push([+vals[0], +vals[1]]);
                            }
                            col++;
                            if (col > xPoints) {
                                col = 0;
                                grid.matrix.push(row);
                                row = [];
                            }
                        }
                    }
                }
                if (grid) {
                    grids.push(grid);
                }
                for (i = 0; i < grids.length; i++) {
                    actions.deferAction('storage:add-object', {objectStoreName: 'grids', object: grids[i]});
                }
                if (redraw) {
                    actions.deferAction('views:draw-input-view');
                    actions.deferAction('views:draw-output-view');
                }
            }).fail(function () {
                alert('Unable to run native command on server.\nPlease check server log output.');
                events.dispatchEvent('views:hide-loading-screens');
            });
        });
    }

    function getAllGrids() {
        return actions.deferAction('storage:get-objects-as-list', {objectStoreName: 'grids'});
    }

    events.addEventListener('grids:preview-button-clicked', function() {
        generateGrids(true).catch(function(e) {
            console.error(e, e.stack);
        });
    });


    actions.handleAction('grids:generate-grids', function(data, deferred) {
        generateGrids(data.redraw).then(deferred.resolve)
        .catch(deferred.reject);
    });

    actions.handleAction('grids:get-all-grids', function(data, deferred) {
        getAllGrids().then(deferred.resolve)
        .catch(deferred.reject);
    });

    actions.handleAction('grids:delete-all-grids', function(data, deferred) {
         deleteAllGrids().then(deferred.resolve)
         .catch(deferred.reject);
    });

})();

(function() {

    var REGION_COLOURS = [
        "#B79762",
        "#004D43",
        "#8FB0FF",
        "#997D87",
        "#5A0007",
        "#809693",
        "#1B4400",
        "#4FC601",
        "#3B5DFF",
        "#4A3B53",
        "#FF2F80",
        "#61615A",
        "#BA0900",
        "#48Afff",
        "#5050FF",
        "#FF34FF",
        "#6fff4b",
        "#FF0BFF",
        "#d70000",
        "#63FFAC"
    ];

    var REGION_DEFAULTS = {
        width: null,
        height: null,
        x: null,
        y: null,
        transformation: 'panoramic',
        params: {
            strength: 1,
            strengthY: 1,
            rotation: 0
        },
        ptz: {
            pan: 0,
            tilt: 0,
            zoom: 1
        },
        roi: {
            x: 0,
            y: 0,
            w: null,
            h: null
        }
    };

    function getActiveRegionIndex() {
        return actions.deferAction('storage:get-object-by-key', {objectStoreName: 'state', key: 'activeRegion'});
    }

    function setActiveRegionIndex(index) {
        return actions.deferAction('storage:update-object-by-key', {objectStoreName: 'state', key: 'activeRegion', object: index});
    }

    function getAllRegions() {
        return actions.deferAction('storage:get-objects-as-list', {objectStoreName: 'regions'});
    }

    function removeAllRegions() {
        removeAllRegionTabs();
        return actions.deferAction('storage:clear-object-store', {objectStoreName: 'regions'});
    }

    function autoArrangeAllRegions() {
        var getOutputDataPromise = actions.deferAction('output:get-output-data');
        var getAllRegionsPromise = getAllRegions();

        actions.deferAction('output:set-image-validity', {validity: false}).then(function() {
            return actions.deferAction('grids:delete-all-grids');
        }).then(function() {
            return Promise.all([getOutputDataPromise, getAllRegionsPromise]);
        }).then(values => {
            var outputRes = values[0];
            var regions = values[1];
            var numberOfRegions = regions.length;
            if (numberOfRegions === 2) {
                regions[0].x = 0;
                regions[0].y = 0;
                regions[0].width = outputRes.width;
                regions[0].height = outputRes.height/2;
                actions.deferAction('storage:update-object-by-index', {objectStoreName: 'regions', index: 0, object: regions[0]});
                refreshRegionInputs(0);
                regions[1].x = 0;
                regions[1].y = outputRes.height/2;
                regions[1].width = outputRes.width;
                regions[1].height = outputRes.height/2;
                actions.deferAction('storage:update-object-by-index', {objectStoreName: 'regions', index: 1, object: regions[1]});
                refreshRegionInputs(1);
            } else {
                var rowHeight = utils.snap(outputRes.height/(Math.ceil(numberOfRegions/2)), 2);
                var columnWidth = utils.snap(outputRes.width/2, 2);
                var row = 0;
                var col = 0;
                var x, y, width, height;
                for (var i = 0; i < numberOfRegions; i++) {
                    if (i === 0 && numberOfRegions % 2 !== 0) {
                        x = 0;
                        y = 0;
                        width = outputRes.width;
                        height = rowHeight;
                        row++;
                    } else {
                        x = col * columnWidth;
                        y = row * rowHeight;
                        width = columnWidth;
                        height = rowHeight;
                        if (col === 1) {
                            col = 0;
                            row++;
                        } else {
                            col++;
                        }
                    }
                    regions[i].x = x;
                    regions[i].y = y;
                    regions[i].width = width;
                    regions[i].height = height;
                    actions.deferAction('storage:update-object-by-index', {objectStoreName: 'regions', index: i, object: regions[i]});
                    refreshRegionInputs(i);
                }
            }
            $('.regions-section .tab')[numberOfRegions - 1].click();
        }).catch(function(e) {
            console.error(e, e.stack);
        });
    }

    function addRegionTab() {
        $('.regions-section .tab-contents').append($('#region-settings-template').clone().html());
        $('.regions-section .tabs').append($('#tab-template').clone().html());
        numberAndColourRegionTabs();
    }

    function loadRegion(index, data, last) {
        addRegionTab();
        setActiveRegionIndex(index).then(function() {
            return actions.deferAction('storage:add-object', {objectStoreName: 'regions', object: REGION_DEFAULTS});
        }).then(function() {
            return setActiveRegionIndex(index);
        }).then(function() {
            var transformation = data.transformation.toLowerCase();
            var regionTabContent = $('.regions-section .tab-content').eq(index);
            regionTabContent.find('.transformation-select').val(transformation);
            showTransformationSection(regionTabContent, transformation);
            regionTabContent.find('.region-x').val(data.position[0]);
            regionTabContent.find('.region-y').val(data.position[1]);
            regionTabContent.find('.region-width').val(data.position[2]);
            regionTabContent.find('.region-height').val(data.position[3]);
            var params = {};
            for (var param in data.param) {
                if (data.param.hasOwnProperty(param)) {
                    regionTabContent.find('[name="' + param + '"]').val(data.param[param]);
                    params[param] = data.param[param];
                }
            }
            regionTabContent.find('[name="pan"]').val(data.ptz[0]);
            regionTabContent.find('[name="tilt"]').val(data.ptz[1]);
            regionTabContent.find('[name="zoom"]').val(data.ptz[2]);
            regionTabContent.find('.input-roi-x').val(data.roi.x);
            regionTabContent.find('.input-roi-y').val(data.roi.y);
            regionTabContent.find('.input-roi-width').val(data.roi.w);
            regionTabContent.find('.input-roi-height').val(data.roi.h);
            var regionData = {
                x: data.position[0],
                y: data.position[1],
                width: data.position[2],
                height: data.position[3],
                transformation: transformation,
                params: params,
                ptz: {
                    pan: data.ptz[0],
                    tilt: data.ptz[1],
                    zoom: data.ptz[2]
                },
                roi: {
                    x: data.roi.x,
                    y: data.roi.y,
                    w: data.roi.w,
                    h: data.roi.h
                }
            };
            return actions.deferAction('storage:update-object-by-index', {objectStoreName: 'regions', index: index, object: regionData});
        }).then(function() {
            if (last) {
                $('.regions-section .tab').eq(index).click();
            }
        });
    }

    function numberAndColourRegionTabs() {
        var tabs = $('.regions-section .tab');
        tabs.each(function(i, tab) {
            tab.querySelector('.tab-text').innerHTML = +i + 1;
            tab.style.backgroundColor = REGION_COLOURS[i];
        });
    }

    function updateRoi(inputData) {
        $('.input-roi-x').val(0);
        $('.input-roi-y').val(0);
        $('.input-roi-width').val(+inputData.width);
        $('.input-roi-height').val(+inputData.height);
        getAllRegions().then(function(regions) {
            for (var i = 0; i < regions.length; i++) {
                var region = regions[i];
                region.roi.x = 0;
                region.roi.y = 0;
                region.roi.w = (+inputData.width);
                region.roi.h = (+inputData.height);
                actions.deferAction('storage:update-object-by-index', {objectStoreName: 'regions', index: i, object: region});
            }
        }).catch(function(e) {
            console.error(e, e.stack);
        });
    }

    actions.handleAction('regions:update-roi', function(data, deferred) {
        updateRoi(data.inputData);
    });

    function addRegion() {
        actions.deferAction('output:clear-output-image').then(function() {
            return actions.deferAction('storage:add-object', {objectStoreName: 'regions', object: REGION_DEFAULTS});
        }).then(function() {
            return getAllRegions();
        }).then(function(regions) {
            return setActiveRegionIndex(regions.length - 1);
        }).then(function() {
            addRegionTab();
            autoArrangeAllRegions();
            actions.deferAction('input:get-input-data').then(function(inputData) {
                updateRoi(inputData);
            });
            return Promise.resolve();
        }).catch(function(e) {
            console.error(e, e.stack);
        });
    }

    function removeRegion(index) {
        return actions.deferAction('storage:remove-object-by-index', {objectStoreName: 'regions', index: index}).then(function() {
            $($('.regions-section .tab')[index]).remove();
            $($('.regions-section .tab-content')[index]).remove();
            autoArrangeAllRegions();
            numberAndColourRegionTabs();
        }).catch(function(e) {
            console.error(e, e.stack);
        });
    }

    function removeAllRegionTabs() {
        $('.regions-section .tabs').empty();
        $('.regions-section .tab-contents').css('border-color', 'grey');
        $('.regions-section .tab-contents').empty();
    }

    function selectRegionTab(tabIndex) {
        setActiveRegionIndex(tabIndex);
        $('.regions-section .tab-contents').css('border-color', REGION_COLOURS[tabIndex]);
        actions.deferAction('views:draw-input-view');
        actions.deferAction('views:draw-output-view');
    }

    function setRegionInputs(index, regionData) {
        var regionEl = $($('.regions-section .tab-content')[index]);
        regionEl.find('.region-x').val(regionData.x);
        regionEl.find('.region-y').val(regionData.y);
        regionEl.find('.region-width').val(regionData.width);
        regionEl.find('.region-height').val(regionData.height);
    }

    function refreshRegionInputs(index) {
        getAllRegions().then(function(regions) {
            var regionData = regions[index];
            setRegionInputs(index, regionData);
        }).catch(function(e) {
            console.error(e, e.stack);
        });
    }

    function setRegionWindow(index, x, y, width, height, redraw) {
        getAllRegions().then(function(regions) {
            var regionData = regions[index];
            regionData.x = x;
            regionData.y = y;
            regionData.width = width;
            regionData.height = height;
            setRegionInputs(index, regionData);
            return actions.deferAction('storage:update-object-by-index', {objectStoreName: 'regions', index: index, object: regionData});
        }).then(function() {
            if (redraw) {
                actions.deferAction('views:draw-output-view');
            }
        }).catch(function(e) {
            console.error(e, e.stack);
        });
    }

    function updateCurrentRegionFromInputs() {
        var outputView = $('.output-view');
        var getActiveRegionIndexPromise = getActiveRegionIndex();
        var getOutputDataPromise = actions.deferAction('output:get-output-data');
        var getConstraintPromise = actions.deferAction('storage:get-objects', {objectStoreName: 'constraint'});
        var getRegionsPromise = getAllRegions();

        actions.deferAction('grids:delete-all-grids').then(function() {
            return Promise.all([getActiveRegionIndexPromise, getOutputDataPromise, getConstraintPromise, getRegionsPromise]).then(values => {
                var index = values[0];
                var outputData = values[1];
                var constraint = values[2];
                var regionsList = values[3];

                var scale = utils.getBestFitScale(outputData.width, outputData.height, outputView.width(), outputView.height());
                var scaledConstraint = {};
                for (var key in constraint) {
                    if (constraint.hasOwnProperty(key)) {
                        scaledConstraint[key] = constraint[key] / scale;
                    }
                }

                var tabContent = $($('.regions-section .tab-content')[index]);
                var paramSection = tabContent.find('.transformation-parameters:visible');
                var regionData = regionsList[index];
                regionData.transformation = paramSection.attr('name');

                regionData.x = +$('.region-x:visible').val();
                regionData.y = +$('.region-y:visible').val();
                regionData.width = +$('.region-width:visible').val();
                regionData.height = +$('.region-height:visible').val();

                regionData.roi.x = +$('.input-roi-x:visible').val();
                regionData.roi.y = +$('.input-roi-y:visible').val();
                regionData.roi.w = +$('.input-roi-width:visible').val();
                regionData.roi.h = +$('.input-roi-height:visible').val();

                regionData.ptz = {};
                var ptzSection = tabContent.find('.ptz');
                var ptzInputs = ptzSection.find('input');
                var ptzInput;
                ptzInputs.each(function(i, el) {
                    ptzInput = $(el);
                    regionData.ptz[ptzInput.attr('name')] = +ptzInput.val();
                });
                if (paramSection.attr('name') !== 'custom') {
                    regionData.params = {};
                    var paramInputs = paramSection.find('input');
                    var paramInput;
                    paramInputs.each(function(i, el) {
                        paramInput = $(el);
                        if (el.type == 'checkbox') {
                            regionData.params[paramInput.attr('name')] = (el.checked) ? 1 : 0;
                        } else {
                            regionData.params[paramInput.attr('name')] = +paramInput.val();
                        }
                    });
                }
                return actions.deferAction('storage:update-object-by-index', {objectStoreName: 'regions', index: index, object: regionData});
            }).then(function() {
                return actions.deferAction('output:set-image-validity', {validity: false});
            }).then(function() {
                actions.deferAction('views:draw-input-view');
            }).then(function() {
                actions.deferAction('views:draw-output-view');
            }).catch(function(e) {
                console.error(e, e.stack);
            });
        });
    }

    function showTransformationSection(regionTabContent, transformationType) {
        var parameters;
        regionTabContent.find('.transformation-parameters').each(function(i, el) {
            parameters = $(el);
            if (parameters.attr('name') === transformationType) {
                parameters.removeClass('hidden');
            } else {
                parameters.addClass('hidden');
            }
        });
    }

    function changeCurrentTransformationType(transformationType) {
        var regionTabContent = $('.regions-section .tab-content:visible');
        showTransformationSection(regionTabContent, transformationType);
        updateCurrentRegionFromInputs();
    }

    function saveCustomConfig(file) {
        var getActiveRegionIndexPromise = getActiveRegionIndex();
        var getAllRegionsPromise = getAllRegions();

        Promise.all([getActiveRegionIndexPromise, getAllRegionsPromise]).then(values => {
            var index = values[0];
            var regionsList = values[1];
            if (file) {
                var reader = new FileReader();
                reader.onload = function(e) {
                    $('.custom-config-file-name:visible').text(file.name);
                    var regionData = regionsList[index];
                    regionData.data = reader.result;
                    actions.deferAction('storage:update-object-by-index', {objectStoreName: 'regions', index: index, object: regionData});
                };
                reader.readAsText(file);
            }
        }).catch(function(e) {
            console.error(e, e.stack);
        });
    }

    function maxConstraintNotification(value, constraint) {
        $('.notification-curtain').removeClass('hidden');
        $('.notification-information').removeClass('hidden');
        $('.notification-information .notification-information-button').focus();
        $('.notification-information .notification-header').text('Value Above Maximum');
        $('.notification-information .notification-content').text('The value provided (' + value + ') was above the maximum. Value was changed to ' + constraint);
    }

    function minConstraintNotification(value, constraint) {
        $('.notification-curtain').removeClass('hidden');
        $('.notification-information').removeClass('hidden');
        $('.notification-information .notification-header').text('Value Below Minimum');
        $('.notification-information .notification-content').text('The value provided (' + value + ') was below the minimum. Value was changed to ' + constraint);
    }

    events.addEventListener('regions:hide-information-notification', function() {
        $('.notification-curtain').addClass('hidden');
        $('.notification-information').addClass('hidden');
    });

    events.addEventListener('regions:add-region', function(eventData, callback) {
        addRegion();
    });

    events.addEventListener('regions:auto-arrange-regions', function(eventData, callback) {
        autoArrangeAllRegions();
    });

    events.addEventListener('regions:remove-region', function(target, callback) {
        var tab = target.closest('.tab');
        removeRegion($('.regions-section .tab').index(tab));
    });

    events.addEventListener('regions:open-custom-file-upload-dialog', function() {
        document.getElementById('custom-config-upload').click();
    });

    events.addEventListener('regions:change-transformation-type', function(target, callback) {
        changeCurrentTransformationType(target.value);
        actions.deferAction('output:set-image-validity', {validity: false}).then(function() {
            return actions.deferAction('views:draw-output-view');
        }).catch(function(e) {
            console.error(e, e.stack);
        });
    });

    events.addEventListener('regions:region-input-value-changed', function(target, callback) {
        var input = $(target);
        var value = +input.val();
        if (input[0].type == 'checkbox') {
            value = (input[0].checked) ? 1 : 0;
            updateCurrentRegionFromInputs();
            return;
        }
        var minAttr = input.attr('data-min');
        if (typeof minAttr !== typeof undefined && minAttr !== false) {
            var min = +input.attr('data-min');
            if (value < min) {
                input.val(min);
                minConstraintNotification(value, min);
            }
        }
        var maxAttr = input.attr('data-max');
        if (typeof maxAttr !== typeof undefined && maxAttr !== false) {
            var max = +input.attr('data-max');
            if (value > max) {
                input.val(max);
                maxConstraintNotification(value, max);
            }
        }
        updateCurrentRegionFromInputs();
    });

    events.addEventListener('regions:set-region-window', function(eventData, callback) {
        setRegionWindow(eventData.index, eventData.x, eventData.y, eventData.width, eventData.height, eventData.redraw);
        actions.deferAction('output:set-image-validity', {validity: false});
    });

    events.addEventListener('regions:upload-custom-config', function(target, callback) {
        saveCustomConfig(target.files[0]);
    });


    actions.handleAction('regions:load-region', function(data, deferred) {
        loadRegion(data.index, data.regionData, data.last);
        return deferred.resolve();
    });

    actions.handleAction('regions:get-region-colours', function(data, deferred) {
        deferred.resolve(REGION_COLOURS);
    });

    actions.handleAction('regions:get-active-region-index', function(data, deferred) {
        getActiveRegionIndex().then(deferred.resolve)
        .catch(deferred.reject);
    });

    actions.handleAction('regions:get-all-regions', function(data, deferred) {
        getAllRegions().then(deferred.resolve)
        .catch(deferred.reject);
    });

    actions.handleAction('regions:get-active-region-index', function(data, deferred) {
        getActiveRegionIndex().then(deferred.resolve)
        .catch(deferred.reject);
    });

    actions.handleAction('regions:set-active-region-index', function(data, deferred) {
        setActiveRegionIndex(data.index).then(deferred.resolve)
        .catch(deferred.reject);
    });

    actions.handleAction('regions:remove-all-regions', function(data, deferred) {
        removeAllRegions().then(deferred.resolve)
        .catch(deferred.reject);
    });


// This is bad TODO Find a way that we don't have to do this.
    events.addEventListener('tabs:select-tab', function(target, callback) {
        if (target.closest('.regions-section')) {
            var index = $('.regions-section .tab').index(target);
            selectRegionTab(index);
        }
    });

})();

(function() {
    function getLayoutJSON() {
        var inputDataPromise = actions.deferAction('input:get-input-data');
        var outputDataPromise = actions.deferAction('output:get-output-data');
        var regionsPromise = actions.deferAction('regions:get-all-regions');
        var settingsPromise = actions.deferAction('settings:get-settings-data');

        return Promise.all([inputDataPromise, outputDataPromise, regionsPromise, settingsPromise]).then(results => {
             var inputData = results[0];
             var outputData = results[1];
             var regions = results[2];
             var settings = results[3];

             var colourspaces = {
                 'planar420': 'yuv',
                 'yuv444': 'yuv',
                 'semiplanar420': 'yuv',
                 'planar444': 'rgb',
                 'luminance': 'grey'
             };

             var colourspace = colourspaces[settings.mode];
             var mode = settings.mode === 'yuv444' ? 'planar444': settings.mode;
             var eccMode = settings.eccMode;

             var data = {
                 inputRes: [+inputData.width, +inputData.height],
                 param:{
                     fov: +inputData.fov,
                     diameter: +inputData.diameter,
                     offsetX: +inputData.offsetX,
                     offsetY: +inputData.offsetY
                 },
                 outputRes: [+outputData.width, +outputData.height],
                 transformations: [],
                 mode: mode,
                 eccMode: eccMode,
                 colourspace: colourspace
             };
             var region;
             for (var i = 0; i < regions.length; i++) {
                 region = regions[i];
                 if (region.transformation !== 'custom') {
                     data.transformations[i] = {
                         transformation: region.transformation[0].toUpperCase() + region.transformation.slice(1),
                         position: [
                             +region.x,
                             +region.y,
                             +region.width,
                             +region.height
                         ],
                         param: region.params,
                         ptz: [
                             +region.ptz.pan,
                             +region.ptz.tilt,
                             +region.ptz.zoom
                         ],
                         roi: region.roi
                     };
                 } else {
                     data.transformations[i] = {
                         transformation: region.transformation[0].toUpperCase() + region.transformation.slice(1),
                         position: [
                             +region.x,
                             +region.y,
                             +region.width,
                             +region.height
                         ],
                         data: region.data,
                         ptz: [
                             +region.ptz.pan,
                             +region.ptz.tilt,
                             +region.ptz.zoom
                         ],
                         roi: region.roi
                     };
                }
            }
            return data;
        });
    }

    function saveLayoutJSON() {
        getLayoutJSON()
            .then(function(layout) {
                var json = JSON.stringify(layout, false, 4);
                var blob = new Blob([json], {type: "application/json"});
                var url = URL.createObjectURL(blob);
                var a = document.createElement('a');
                a.download = "layout.json";
                a.href = url;
                document.body.appendChild(a);
                a.click();
                document.body.removeChild(a);
            });
    }

    function loadLayoutToGUI(layoutData) {
        var removeAllRegionsPromise = actions.deferAction('regions:remove-all-regions');
        var removeAllGridsPromise = actions.deferAction('grids:delete-all-grids');
        var setOutputResolution = actions.deferAction('output:set-output-resolution', {width: layoutData.outputRes[0], height: layoutData.outputRes[1]});
        actions.deferAction('settings:set-settings-data', {settingsData: {mode: layoutData.mode, eccMode: layoutData.eccMode}});
        document.querySelector('.settings-mode').value = layoutData.mode;
        document.querySelector('.settings-ecc-mode').value = layoutData.eccMode;

        Promise.all([removeAllRegionsPromise, removeAllGridsPromise])
        .then(function() {
            return actions.deferAction('input:get-input-data');
        }).then(function(inputData) {
            if (+inputData.width !== +layoutData.inputRes[0] || +inputData.height !== +layoutData.inputRes[1]) {
                actions.deferAction('notifications:input-notification', {text: 'Layout does not match current image resolution. Image data not loaded.'});
                return Promise.resolve();
            } else {
                var inputParameters = {
                    name: inputData.name,
                    width: layoutData.inputRes[0],
                    height: layoutData.inputRes[1],
                    fov: layoutData.param.fov,
                    diameter: layoutData.param.diameter,
                    offsetX: layoutData.param.offsetX,
                    offsetY: layoutData.param.offsetY
                };
                return actions.deferAction('input:set-input-data', {inputData: inputParameters});
            }
        }).then(setOutputResolution)
        .then(function() {
            var lastTab = false;
            for (var i = 0; i < layoutData.transformations.length; i++) {
                if (i === layoutData.transformations.length - 1) {
                    lastTab = true;
                }
                actions.deferAction('regions:load-region', {index: i, regionData: layoutData.transformations[i], last: lastTab});
            }
        });
    }

    function loadLayout(layoutData) {
        actions.deferAction('input:get-input-data').then(function(inputData) {
            if (+inputData.width !== +layoutData.inputRes[0] || +inputData.height !== +layoutData.inputRes[1]) {
                $('.notification-curtain').removeClass('hidden');
                $('.notification-confirmation').removeClass('hidden');
                $('.notification-header').text('Layout Issue');
                $('.notification-content').text('The layout does not match the current image resolution. Input parameters will not be loaded.');

                var handler = function() {
                    $('.notification-curtain').addClass('hidden');
                    $('.notification-confirmation').addClass('hidden');
                    loadLayoutToGUI(layoutData);
                };

                $('.notification-confirmation-button').unbind();
                $('.notification-confirmation-button').bind('click', handler);

            } else {
                loadLayoutToGUI(layoutData);
            }
        });
    }

    events.addEventListener('layout:cancel-load-layout-button', function() {
        $('.notification-curtain').addClass('hidden');
        $('.notification-confirmation').addClass('hidden');
    });

    actions.handleAction('layout:get-layout', function(data, deferred) {
        getLayoutJSON().then(deferred.resolve)
        .then(deferred.reject);
    });
    events.addEventListener('layout:save-layout', function(eventData, callback) {
        saveLayoutJSON();
    });
    events.addEventListener('layout:open-load-layout-dialog', function() {
        $('#load-layout-input').click();
    });
    events.addEventListener('layout:load-layout', function(eventData, callback) {
        var fileReader = new FileReader();
        fileReader.onload = function(event) {
            loadLayout(JSON.parse(event.target.result));
        };
        fileReader.readAsText(eventData.files[0]);
    });

})();

(function() {

    var draw = new Draw();

    var views = {
        input: draw.svg({class: 'input'}),
        output: draw.svg({class: 'output'})
    };

    document.querySelector('.input-view-drawing-canvas').appendChild(views.input);
    document.querySelector('.output-view-drawing-canvas').appendChild(views.output);

    function clearView(viewName) {
        while (views[viewName].firstChild) {
            views[viewName].removeChild(views[viewName].firstChild);
        }
    }

    var resizing = false;
    var currentActiveDragPoint, currentActiveRegion, selectBox, currentScale, currentIndex, currentConstraints;

    var drag = function(event) {
        if (resizing) {
            if (currentActiveDragPoint) {

                actions.deferAction('grids:delete-all-grids').then(function() {
                    currentActiveRegion.setAttribute('fill', 'none');
                    currentActiveRegion.setAttribute('stroke', 'none');
                });
                $('g.grid').each(function() {
                    this.remove();
                });

                var top = document.querySelector('.drag-point[data-drag-type="top"]');
                var bottom = document.querySelector('.drag-point[data-drag-type="bottom"]');
                var left = document.querySelector('.drag-point[data-drag-type="left"]');
                var right = document.querySelector('.drag-point[data-drag-type="right"]');

                var topX = +top.getAttribute('cx');
                var topY = +top.getAttribute('cy');
                var bottomX = +bottom.getAttribute('cx');
                var bottomY = +bottom.getAttribute('cy');
                var leftX = +left.getAttribute('cx');
                var leftY = +left.getAttribute('cy');
                var rightX = +right.getAttribute('cx');
                var rightY = +right.getAttribute('cy');

                var clippingRect = document.getElementById('region-select-clipper').querySelector('rect');

                if (currentActiveDragPoint.getAttribute('data-drag-type') === 'top') {
                    top.setAttribute('cy', Math.max(currentConstraints.minY, Math.min(currentConstraints.maxY, event.offsetY)));
                    left.setAttribute('cy', topY + ((bottomY - topY)/2));
                    right.setAttribute('cy', topY + ((bottomY - topY)/2));

                    selectBox.setAttribute('y', top.getAttribute('cy'));
                    selectBox.setAttribute('height', bottom.getAttribute('cy') - top.getAttribute('cy'));
                    clippingRect.setAttribute('y', top.getAttribute('cy'));
                    clippingRect.setAttribute('height', bottom.getAttribute('cy') - top.getAttribute('cy'));
                }
                else if (currentActiveDragPoint.getAttribute('data-drag-type') === 'bottom') {
                    bottom.setAttribute('cy', Math.max(currentConstraints.minY, Math.min(currentConstraints.maxY, event.offsetY)));
                    left.setAttribute('cy', topY + ((bottomY - topY)/2));
                    right.setAttribute('cy', topY + ((bottomY - topY)/2));

                    selectBox.setAttribute('y', top.getAttribute('cy'));
                    selectBox.setAttribute('height', bottom.getAttribute('cy') - top.getAttribute('cy'));
                    clippingRect.setAttribute('y', top.getAttribute('cy'));
                    clippingRect.setAttribute('height', bottom.getAttribute('cy') - top.getAttribute('cy'));
                }
                else if (currentActiveDragPoint.getAttribute('data-drag-type') === 'left') {
                    left.setAttribute('cx', Math.max(currentConstraints.minX, Math.min(currentConstraints.maxX, event.offsetX)));
                    top.setAttribute('cx', leftX + ((rightX - leftX)/2));
                    bottom.setAttribute('cx', leftX + ((rightX - leftX)/2));

                    selectBox.setAttribute('x', left.getAttribute('cx'));
                    selectBox.setAttribute('width', right.getAttribute('cx') - left.getAttribute('cx'));
                    clippingRect.setAttribute('x', left.getAttribute('cx'));
                    clippingRect.setAttribute('width', right.getAttribute('cx') - left.getAttribute('cx'));
                }
                else if (currentActiveDragPoint.getAttribute('data-drag-type') === 'right') {
                    right.setAttribute('cx', Math.max(currentConstraints.minX, Math.min(currentConstraints.maxX, event.offsetX)));
                    top.setAttribute('cx', leftX + ((rightX - leftX)/2));
                    bottom.setAttribute('cx', leftX + ((rightX - leftX)/2));

                    selectBox.setAttribute('x', left.getAttribute('cx'));
                    selectBox.setAttribute('width', right.getAttribute('cx') - left.getAttribute('cx'));
                    clippingRect.setAttribute('x', left.getAttribute('cx'));
                    clippingRect.setAttribute('width', right.getAttribute('cx') - left.getAttribute('cx'));
                }
            }
        }
    };

    document.querySelector('.output-view-drawing-canvas').addEventListener('mousemove', drag);
    document.addEventListener('mouseup', function() {
        if (resizing) {
            var dragPoints = document.querySelectorAll('.drag-point');
            for (var i = 0; i < dragPoints.length; i++) {
                dragPoints[i].setAttribute('fill', 'orange');
            }
            var top = document.querySelector('.drag-point[data-drag-type="top"]');
            var bottom = document.querySelector('.drag-point[data-drag-type="bottom"]');
            var left = document.querySelector('.drag-point[data-drag-type="left"]');
            var right = document.querySelector('.drag-point[data-drag-type="right"]');
            events.dispatchEvent('regions:set-region-window', {
                index: currentIndex,
                width: utils.snap((right.getAttribute('cx') - left.getAttribute('cx')) / currentScale, 2),
                height: utils.snap(( bottom.getAttribute('cy') - top.getAttribute('cy')) / currentScale, 2),
                x: utils.snap(+left.getAttribute('cx') / currentScale, 2),
                y: utils.snap(+top.getAttribute('cy') / currentScale, 2),
                redraw: true
            });
            resizing = false;
        }
    });

    function drawInputView() {
        clearView('input');
        var inputView = $('.input-view');
        return actions.deferAction('input:get-input-data').then(function(inputData) {
            var scale = utils.getBestFitScale(inputData.width, inputData.height, inputView.width(), inputView.height());
            if (scale !== Infinity) {
                var width = inputData.width * scale;
                var height = inputData.height * scale;
                var svgDrawingArea = $('.input-view-drawing-canvas');
                svgDrawingArea.width(width);
                svgDrawingArea.height(height);
                views.input.setAttribute('width', width);
                views.input.setAttribute('height', height);
                var img = draw.image(inputData.image, {width: width, height: height});
                views.input.appendChild(img);
                return Promise.resolve();
            }
        });
    }

    function linkGridBoxes(inRect, outRect) {
        outRect.addEventListener('mouseover', function() {
            outRect.setAttribute('opacity', 0.3);
            inRect.setAttribute('fill', 'white');
            inRect.setAttribute('opacity', 0.5);
            inRect.setAttribute('stroke', 'red');
            inRect.setAttribute('stroke-opacity', 1);
        });
        outRect.addEventListener('mouseout', function() {
            outRect.setAttribute('opacity', 0);
            inRect.setAttribute('fill', 'none');
            inRect.setAttribute('opacity', 1);
            inRect.setAttribute('stroke', 'white');
            inRect.setAttribute('stroke-opacity', 0.5);
        });
    }

    function drawGrid(gridData, regionWindow, scale) {
        var inputView = $('.input-view');
        actions.deferAction('input:get-input-data').then(function(inputData) {
            var inputScale = utils.getBestFitScale(inputData.width, inputData.height, inputView.width(), inputView.height());
            var outputGrid = draw.group({class: 'grid'});
            var inputGrid = draw.group({class: 'grid'});
            var xStep = gridData.xStep * scale;
            var yStep = gridData.yStep * scale;
            var i, j;
            if (gridData.valid) {
                var previousLine = utils.scaleLine(gridData.matrix[0], inputScale);
                var currentLine;
                var outX = 0;
                var outY = 0;
                var regionWidth, regionHeight, regionX, regionY;
                for (i = 1; i < gridData.matrix.length; i++) {
                    currentLine = utils.scaleLine(gridData.matrix[i], inputScale);
                    for (j = 0; j < currentLine.length - 1; j++) {
                        regionWidth = regionWindow.getAttribute('width');
                        regionHeight = regionWindow.getAttribute('height');
                        regionX = +regionWindow.getAttribute('x');
                        regionY = +regionWindow.getAttribute('y');
                        var outRect = draw.rect({
                            x: outX + regionX,
                            y: outY + regionY,
                            width: outX + xStep > regionWidth ? regionWidth - outX: xStep,
                            height: outY + yStep > regionHeight ? regionHeight - outY: yStep,
                            fill: 'white',
                            opacity: 0,
                            stroke: 'red'
                        });
                        var inRect = draw.polygon([previousLine[j], previousLine[j + 1], currentLine[j + 1], currentLine[j]], {fill: 'none', stroke: 'white', 'stroke-opacity': 0.5, opacity: 1});
                        inputGrid.appendChild(inRect);
                        outputGrid.appendChild(outRect);
                        outX += xStep;
                        linkGridBoxes(inRect, outRect);
                    }
                    outX = 0;
                    outY += yStep;
                    previousLine = currentLine;
                }
            } else {
                actions.deferAction('notifications:input-notification', {text: 'One or more of the grids has points that fall outside of the view.'});
                var circle;
                for (i = 0; i < gridData.matrix.length; i++) {
                    for (j = 0; j < gridData.matrix[i].length; j++) {
                        circle = draw.circle({r: 2, fill: 'white', stroke: 'black', cx: gridData.matrix[i][j][0] * inputScale, cy: gridData.matrix[i][j][1] * inputScale});
                        inputGrid.appendChild(circle);
                    }
                }
            }
            views.input.appendChild(inputGrid);
            views.output.insertBefore(outputGrid, selectBox);
            hideLoadingScreens();
        }).catch(function(e) {
            console.error(e, e.stack);
        });
    }

    function drawGridOutline(gridData, index) {
        var inputView = $('.input-view');
        var getInputDataPromise = actions.deferAction('input:get-input-data');
        var getRegionColours = actions.deferAction('regions:get-region-colours');

        Promise.all([getInputDataPromise, getRegionColours]).then(values => {
            var inputData = values[0];
            var regionColours = values[1];
            var inputScale = utils.getBestFitScale(inputData.width, inputData.height, inputView.width(), inputView.height());
            var inputGrid = draw.group({class: 'grid'});
            var top = gridData.matrix[0];
            var bottom = gridData.matrix[gridData.matrix.length - 1];
            var left = [];
            var right = [];
            gridData.matrix.map(function(x) {
                left.push(x[0]);
                right.push(x[x.length - 1]);
            });

            var line1 = draw.polyline(utils.scaleLine(top, inputScale), {'stroke-width': 3, stroke: regionColours[index], fill: 'none'});
            var line2 = draw.polyline(utils.scaleLine(right, inputScale), {'stroke-width': 3, stroke: regionColours[index], fill: 'none'});
            var line3 = draw.polyline(utils.scaleLine(bottom, inputScale), {'stroke-width': 3, stroke: regionColours[index], fill: 'none'});
            var line4 = draw.polyline(utils.scaleLine(left, inputScale), {'stroke-width': 3, stroke: regionColours[index], fill: 'none'});
            inputGrid.appendChild(line1);
            inputGrid.appendChild(line2);
            inputGrid.appendChild(line3);
            inputGrid.appendChild(line4);
            views.input.appendChild(inputGrid);
        }).catch(function(e) {
            console.error(e, e.stack);
        });
    }

    function drawOutputView() {
        clearView('output');
        var outputView = $('.output-view');
        var defs = views.output.querySelector('defs');
        if (!defs) {
            defs = draw.defs();
            views.output.insertBefore(defs, views.output.firstChild);
        }

        var getOutputDataPromise = actions.deferAction('output:get-output-data');
        var getActiveRegionIndexPromise = actions.deferAction('regions:get-active-region-index');
        var getAllRegionsPromise = actions.deferAction('regions:get-all-regions');
        var getAllGridsPromise = actions.deferAction('grids:get-all-grids');
        var getRegionColoursPromise = actions.deferAction('regions:get-region-colours');

        return Promise.all([getOutputDataPromise, getActiveRegionIndexPromise, getAllRegionsPromise, getAllGridsPromise, getRegionColoursPromise]).then(values => {
            var outputData = values[0];
            var index = values[1];
            var regions = values[2];
            var grids = values[3];
            var regionColours = values[4];

            var scale = utils.getBestFitScale(outputData.width, outputData.height, outputView.width(), outputView.height());
            var outputWidth = outputData.width * scale;
            var outputHeight = outputData.height * scale;
            var svgDrawingArea = $('.output-view-drawing-canvas');
            svgDrawingArea.width(outputWidth);
            svgDrawingArea.height(outputHeight);
            views.output.setAttribute('width', outputWidth);
            views.output.setAttribute('height', outputHeight);

            var hlines = [];
            var vlines = [];
            var activeRegion;
            var activeIndex;
            for (var i = 0; i < regions.length; i++) {
                var width = scale * regions[i].width;
                var height = scale * regions[i].height;
                var x = scale * regions[i].x;
                var y = scale * regions[i].y;
                var regionWindow = draw.rect({
                    width: width,
                    height: height,
                    fill: outputData.image === null ?  regionColours[i]: 'none',
                    'fill-opacity': 0.5,
                    stroke: regionColours[i],
                    'stroke-width': 6,
                    'stroke-opacity': 1,
                    x: x,
                    y: y
                });
                views.output.appendChild(regionWindow);
                if (i === index) {
                    activeRegion = regionWindow;
                    activeIndex = i;
                } else {
                    hlines.push({y: y, x: x, x2: x + width});
                    hlines.push({y: y + height, x: x, x2: x + width});
                    vlines.push({x: x, y: y, y2: y + height});
                    vlines.push({x: x + width, y: y, y2: y + height});
                }

                if (grids[i] && grids[i].valid) {
                    drawGridOutline(grids[i], i);
                }

                var clipper = draw.clipPath({id: 'region-clipper-' + i});
                var clipRect =  draw.rect({width: width, height: height, x: x, y: y});
                clipper.appendChild(clipRect);
                defs.appendChild(clipper);
                regionWindow.setAttribute('clip-path', 'url(#region-clipper-' + i + ')');
            }

            if (activeRegion) {
                if (grids[activeIndex]) {
                    drawGrid(grids[activeIndex], activeRegion, scale);
                }
                activeRegion.setAttribute('pointer-events', 'none');
                activeRegion.setAttribute('fill', 'none');

                var constraint = {
                    minX: 0,
                    minY: 0,
                    maxX: outputWidth,
                    maxY: outputHeight
                };

                function valueIsBetween(n, v1, v2) {
                    return (n > v1) && (n < v2);
                }

                function linesConflictOnAxis(p1, p2, p3, p4) {
                    return ((p1 === p3) && (p2 === p4)) || valueIsBetween(p1, p3, p4) || valueIsBetween(p2, p3, p4) || valueIsBetween(p3, p1, p2) || valueIsBetween(p4, p1, p2);
                }

                var regionWidth = activeRegion.getAttribute('width');
                var regionHeight = activeRegion.getAttribute('height');

                x = activeRegion.getAttribute('x');
                var x2 = x + regionWidth;
                y = activeRegion.getAttribute('y');
                var y2 = y + regionHeight;

                var line;
                for (var j = 0; j < hlines.length; j++) {
                    line = hlines[j];
                    if (line.y <= y) {
                        if (linesConflictOnAxis(line.x, line.x2, x, x2)) {
                            if (line.y > constraint.minY) {
                                constraint.minY = line.y;
                            }
                        }
                    } else if (line.y >= y2) {
                        if (linesConflictOnAxis(line.x, line.x2, x, x2)) {
                            if (line.y < constraint.maxY) {
                                constraint.maxY = line.y;
                            }
                        }
                    }
                }
                for (j = 0; j < vlines.length; j++) {
                    line = vlines[j];
                    if (line.x <= x) {
                        if (linesConflictOnAxis(line.y, line.y2, y, y2)) {
                            if (line.x > constraint.minX) {
                                constraint.minX = line.x;
                            }
                        }
                    } else if (line.x >= x2) {
                        if (linesConflictOnAxis(line.y, line.y2, y, y2)) {
                            if (line.x < constraint.maxX) {
                                constraint.maxX = line.x;
                            }
                        }
                    }
                }

                var regionX = +$('.region-x').eq(activeIndex).val();
                var regionY = +$('.region-y').eq(activeIndex).val();
                regionWidth = +$('.region-width').eq(activeIndex).val();
                regionHeight = +$('.region-height').eq(activeIndex).val();

                $('.region-x').eq(activeIndex).attr('data-min', constraint.minX / scale);
                $('.region-x').eq(activeIndex).attr('data-max', constraint.maxX / scale - regionWidth);
                $('.region-y').eq(activeIndex).attr('data-min', constraint.minY / scale);
                $('.region-y').eq(activeIndex).attr('data-max', constraint.maxY / scale - regionHeight);
                $('.region-width').eq(activeIndex).attr('data-min', 2);
                $('.region-width').eq(activeIndex).attr('data-max', (constraint.maxX / scale) - regionX);
                $('.region-height').eq(activeIndex).attr('data-min', 2);
                $('.region-height').eq(activeIndex).attr('data-max', (constraint.maxY / scale - regionY));

                var createDragPoint = function(x, y, type) {
                    var dragPoint = draw.circle({class: 'drag-point', r: 10, cx: x, cy: y, fill: 'orange', stroke: 'black', 'data-drag-type': type});
                    dragPoint.addEventListener('mousedown', function(event) {
                        dragPoint.setAttribute('fill', 'blue');
                        resizing = true;
                        currentActiveDragPoint = dragPoint;
                        currentActiveRegion = activeRegion;
                    });
                    views.output.appendChild(dragPoint);
                };

                selectBox = draw.rect({class: 'select-box', width: regionWidth * scale, height: regionHeight * scale, x: regionX * scale, y: regionY * scale, fill: 'none', stroke: 'orange', 'stroke-width': 6, 'stroke-dasharray': '5 5'});
                views.output.appendChild(selectBox);

                var selectClipper = draw.clipPath({id: 'region-select-clipper'});
                var selectClipRect =  draw.rect({width: regionWidth * scale, height: regionHeight * scale, x: regionX * scale, y: regionY * scale});
                selectClipper.appendChild(selectClipRect);
                defs.appendChild(selectClipper);
                selectBox.setAttribute('clip-path', 'url(#region-select-clipper)');

                var dragTop = createDragPoint((regionX + (regionWidth/2)) * scale, regionY * scale, 'top');
                var dragBottom = createDragPoint((regionX + (regionWidth/2)) * scale, (regionY + regionHeight) * scale, 'bottom');
                var dragLeft = createDragPoint(regionX * scale, (regionY + (regionHeight/2)) * scale, 'left');
                var dragRight = createDragPoint((regionX + regionWidth) * scale, (regionY + (regionHeight/2)) * scale, 'right');

                currentScale = scale;
                currentIndex = index;
                currentConstraints = constraint;

                if (outputData.image !== null) {
                    var img = draw.image(outputData.image, {width: outputWidth, height: outputHeight, class: 'output-image'});
                    views.output.insertBefore(img, views.output.firstChild);

                    if (!outputData.imageIsValid) {
                        img.setAttribute('opacity', 0.5);
                    }
                }
            }
            return Promise.resolve();
        });
    }

    function showLoadingScreens() {
        $('.loading').each(function(i, el) {
            el.classList.remove('hidden');
        });
    }

    function hideLoadingScreens() {
        $('.loading').each(function(i, el) {
            el.classList.add('hidden');
        });
    }

    function saveInputViewAsImage() {
        var svg = document.querySelector('.input-view-drawing-canvas svg');
        var canvas = document.createElement('canvas');
        canvas.width = svg.getAttribute('width');
        canvas.height = svg.getAttribute('height');
        var ctx = canvas.getContext('2d');
        var data = (new XMLSerializer()).serializeToString(svg);
        var DOMURL = window.URL || window.webkitURL || window;

        var img = new Image();
        var svgBlob = new Blob([data], {type: 'image/svg+xml;charset=utf-8'});
        var url = DOMURL.createObjectURL(svgBlob);

        img.onload = function () {
            ctx.drawImage(img, 0, 0);
            DOMURL.revokeObjectURL(url);
            var imgURI = canvas
                .toDataURL('image/png')
                .replace('image/png', 'image/octet-stream');

            var a = document.createElement('a');
            a.setAttribute('download', 'inputView.png');
            a.setAttribute('href', imgURI);
            document.body.appendChild(a);
            a.click();
            document.body.removeChild(a);
        };
        img.src = url;
    }

    events.addEventListener('views:show-loading-screens', showLoadingScreens);
    events.addEventListener('views:hide-loading-screens', hideLoadingScreens);
    events.addEventListener('view:save-input-view-image', saveInputViewAsImage);

    actions.handleAction('views:draw-input-view', function(data, deferred) {
        drawInputView().then(deferred.resolve)
        .catch(deferred.reject);
    });
    actions.handleAction('views:draw-output-view', function(data, deferred) {
        drawOutputView().then(deferred.resolve)
        .catch(deferred.reject);
    });

    window.onresize = utils.throttle(function() {
        drawInputView();
        drawOutputView();
    }, 100);

})();

// Used conversion numbers from http://www.equasys.de/colorconversion.html
(function() {

    let getWorker = function(workerType, deferred) {
        let worker = new Worker(`js/workers/${workerType}.js`);
        worker.onmessage = function(e) {
            worker.terminate();
            deferred.resolve(e.data);
        };
        worker.onerror = function(e) {
            deferred.reject(e);
        };
        return worker
    };

    actions.handleAction('convert:to-rgba', function(data, deferred) {
        let worker = getWorker('rgba', deferred);
        worker.postMessage(data.buffer, [data.buffer]);
    });

    actions.handleAction('convert:to-yuv', function(data, deferred) {
        let worker = getWorker('yuv', deferred);
        worker.postMessage(data.buffer, [data.buffer]);
    });


})();

/**
 * A class for sending ajax requests.
 * @constructor
 */

const AjaxModule = function() {

    /**
     * creates a promise based onload handler to handle the result of an http request.
     * @param {XMLHttpRequest} httpRequest - The request that the handler is being created for.
     * @param resolve - the resolve passed down from the request's promise.
     * @param reject - the resolve passed down from the request's promise.
     * @returns {function()} - the onload handler.
     */
    const createOnLoadHandler = (httpRequest, resolve, reject, responseType) => {
        return () => {
            if (httpRequest.status >= 200 && httpRequest.status < 300) {
                if (responseType === 'xml') {
                    if (httpRequest.responseXML) {
                       resolve(httpRequest.responseXML);
                    } else {
                       reject(new DOMParser().parseFromString(httpRequest.response, 'application/xml'));
                    }
                }
                else {
                    resolve(httpRequest.response);
                }
            } else {
                reject({
                    status: httpRequest.status,
                    statusText: httpRequest.statusText
                });
            }
        };
    };

    /**
     *
     * creates a promise based onerror handler to handle the result of an http request.
     * @param {XMLHttpRequest} httpRequest - The request that the handler is being created for.
     * @param reject - the reject passed down from the request's promise.
     * @returns {function()} - the onerror handler.
     */
    const createOnErrorHandler = (httpRequest, reject) => {
        return () => {
            reject({
                status: 'No Status',
                statusText: 'Connection refused',
                response: 'No response - Please check your connection to the server.'
            });
        };
    };

    /**
     * Adds the given headers to the http request
     * @param {XMLHttpRequest} httpRequest - the request to add the headers to.
     * @param headers - the headers to add to the request.
     */
    const writeHeaders = (httpRequest, headers) => {
        Object.keys(headers).forEach((key) => {
            httpRequest.setRequestHeader(key, headers[key]);
        });
    };

    /**
     * Creates and adds the handlers to the http request.
     * @param {XMLHttpRequest} httpRequest - the request to add the headers to.
     * @param resolve
     * @param reject
     */
    const addHandlers = (httpRequest, resolve, reject, responseType) => {
        httpRequest.onload = createOnLoadHandler(httpRequest, resolve, reject, responseType);
        httpRequest.onerror = createOnErrorHandler(httpRequest, reject);
    };

    /**
     * Creates a GET request
     * @param {String} url - the url for the request
     * @param {Object} headers - the headers to add to the request
     * @param {String} responseType - either left null or set to "arraybuffer"
     */
    this.get = (url, headers, responseType) => {
        return this.request({url: url, method: 'GET', headers: headers, responseType: responseType});
    };

    /**
     * Creates a POST request
     * @param {String} url - the url for the request
     * @param {*} data - the data to add to the request
     * @param {Object} headers - the headers to add to the request
     * @param {String} responseType - either left null or set to "arraybuffer"
     */
    this.post = (url, data, headers, responseType) => {
        return this.request({url: url, method: 'POST', data: data, headers: headers, responseType: responseType});
    };

    /**
     * Creates a PUT request
     * @param {String} url - the url for the request
     * @param {*} data - the data to add to the request
     * @param {Object} headers - the headers to add to the request
     */
    this.put = (url, data, headers) => {
        return this.request({url: url, method: 'PUT', data: data, headers: headers});
    };

    this.delete = (url, headers) => {
        return this.request({url: url, method: 'DELETE', headers: headers});
    };

    /**
     * Creates an XMLHttpRequest with the given options
     * @param {Object} options - the options to create the request with.
     */
    this.request = (options) => {
        return new Promise((resolve, reject) => {
            let httpRequest = new XMLHttpRequest();
            let headers = {};
            if (options.headers) {
                headers = options.headers;
            }
            if (options.responseType === 'arraybuffer') {
                httpRequest.responseType = options.responseType;
            }
            httpRequest.open(options.method, options.url);
            addHandlers(httpRequest, resolve, reject, options.responseType);
            writeHeaders(httpRequest, headers);
            if (options.data) {
                let data = options.data;
                httpRequest.send(data);
            } else {
                httpRequest.send();
            }
        });
    };

};

const ajax = new AjaxModule();


(function() {
    function downloadBinary() {
        actions.deferAction('layout:get-layout').then(function(layout) {
            ajax.post('/generate-binary', JSON.stringify(layout), {}, 'arraybuffer')
               .then(function(bin) {
                    let arr = new Uint8Array(bin);
                    let blob = new Blob([arr]);
                    var url = URL.createObjectURL(blob);
                    var dlAnchorElem = document.getElementById('download-anchor');
                    dlAnchorElem.setAttribute("href", url);
                    dlAnchorElem.setAttribute("download", "config.bin");
                    dlAnchorElem.click();
                    setTimeout(function(){
                        window.URL.revokeObjectURL(url);
                    }, 100);
                }).fail(function () {
                    alert('Unable to run native command on server.\nPlease check server log output.');
                });
        });
    }
    events.addEventListener('binary:download-binary', function() {
        downloadBinary();
    });
})();

(function () {
    $.ajaxTransport("+binary", function (options, originalOptions, jqXHR) {
        if (checkOptions(options)) {
            return {
                send: function (headers, callback) {
                    var url = options.url;
                    var type = options.type;
                    var async = options.async || true;
                    var dataType = options.responseType || "blob";
                    var data = options.data || null;

                    var xmlRequest = new XMLHttpRequest();

                    xmlRequest.addEventListener('load', function () {
                        var data = {};
                        data[options.dataType] = xmlRequest.response;
                        callback(xmlRequest.status, xmlRequest.statusText, data, xmlRequest.getAllResponseHeaders());
                    });
                    xmlRequest.open(type, url, async);
                    processHeaders(headers, xmlRequest);
                    xmlRequest.responseType = dataType;
                    xmlRequest.send(data);

                    function processHeaders(headers, xhr) {
                        for (var i in headers) {
                            if (headers.hasOwnProperty(i)) {
                                xhr.setRequestHeader(i, headers[i]);
                            }
                        }
                    }
                },
                abort: function () {
                    jqXHR.abort();
                }
            };

            
        }

        function checkOptions(options) {
            return window.FormData && ((options.dataType && (options.dataType === 'binary')) || (options.data && ((window.ArrayBuffer && options.data instanceof ArrayBuffer) || (window.Blob && options.data instanceof Blob))));
        }
    });

    function getInputDataBuffer(inputData) {
        return new Promise((resolve, reject) => {
            let canvas = document.createElement('canvas');
            canvas.width = inputData.width;
            canvas.height = inputData.height;
            let context = canvas.getContext('2d');
            let img = new Image(inputData.width, inputData.height);
            img.onload = function () {
                context.drawImage(img, 0, 0);
                let pixels = context.getImageData(0, 0, canvas.width, canvas.height);
                resolve(pixels.data.buffer);
            };
            img.onerror = function(e) {
                reject(e);
            };
            img.src = inputData.image;
        })
    }

    function transformImage(imageData) {
        return new Promise((resolve, reject) => {
            $.ajax({
                url: '/transform-image',
                type: 'POST',
                dataType: "binary",
                contentType: 'application/octet-stream',
                responseType: 'arraybuffer',
                data: imageData,
                processData: false
            }).done(function (result) {
                resolve(result)
            }).fail(function () {
                alert('Unable to run native command on server.\nPlease check server log output.');
                reject();
                events.dispatchEvent('views:hide-loading-screens');
            });
        });
    }

    function applyTransformations() {
        var getInputDataPromise = actions.deferAction('input:get-input-data');
        var getSettingsDataPromise = actions.deferAction('settings:get-settings-data');
        var generateGridsPromise = actions.deferAction('grids:generate-grids', {redraw: false});
        return Promise.all([getInputDataPromise, getSettingsDataPromise, generateGridsPromise]).then(values => {
            var inputData = values[0];
            var settingsData = values[1];
            return getInputDataBuffer(inputData).then(function(buffer) {
                var newImageData = new Uint8Array(buffer);
                var transformImagePromise = transformImage(newImageData);
                var getOutputDataPromise = actions.deferAction('output:get-output-data');
                return Promise.all([transformImagePromise, getOutputDataPromise]);
            }).then(values => {
                var rgbaData = values[0];
                var outputData = values[1];
                var outputCanvas = document.createElement('canvas');
                outputCanvas.width = outputData.width;
                outputCanvas.height = outputData.height;
                var outputContext = outputCanvas.getContext('2d');
                var outputPixels = outputContext.createImageData(outputCanvas.width, outputCanvas.height);
                var newData = new Uint8Array(rgbaData);
                for (var i = 0; i < outputPixels.data.length; i++) {
                    outputPixels.data[i] = newData[i];
                }
                outputContext.putImageData(outputPixels, 0, 0);
                var deferred = new Defer();
                outputCanvas.toBlob(function (blob) {
                    deferred.resolve(URL.createObjectURL(blob));
                });
                return deferred.promise;
            }).then(function(objectURL) {
                return actions.deferAction('output:set-output-image', {url: objectURL});
            }).then(function() {
                return actions.deferAction('views:draw-input-view');
            }).then(function() {
                return actions.deferAction('views:draw-output-view');
            });
        }).catch(function(e) {
            console.error(e, e.stack);
        });
    }

    events.addEventListener('transform:apply-transformations', function() {
        applyTransformations();
    });

   
})();

(function() {

    function getSettingsData() {
        return actions.deferAction('storage:get-objects', {objectStoreName: 'settings'});
    }

    function setSettingsControls(settingsData) {
        $('.settings-mode').val(settingsData.mode);
        $('.settings-ecc-mode').val(settingsData.eccMode);
        storeSettingsValues(settingsData);
    }

    function getSettingsControlValues() {
        return {
            mode: $('.settings-mode').val(),
            eccMode: $('.settings-ecc-mode').val(),
        };
    }

    function storeSettingsValues(settingsData) {
        return actions.deferAction('storage:update-objects', {objectStoreName: 'settings', objects: settingsData}).then(function() {
            return actions.deferAction('views:draw-input-view');
        });
    }

    function aboutNotification() {
        $('.notification-curtain').removeClass('hidden');
        $('.about').removeClass('hidden');
    }

    events.addEventListener('settings:user-input-changed', function() {
        storeSettingsValues(getSettingsControlValues());
    });

    actions.handleAction('settings:set-settings-data', function(data, deferred) {
        storeSettingsValues(data.settingsData).then(deferred.resolve)
        .catch(deferred.reject);
    });

    actions.handleAction('settings:get-settings-data', function(data, deferred) {
        getSettingsData().then(deferred.resolve)
        .catch(deferred.reject);
    });

    actions.handleAction('settings:about-gdc-tool', function(data, deferred) {
        aboutNotification();
    });

    actions.handleAction('settings:hide-about', function(data, deferred) {
        $('.notification-curtain').addClass('hidden');
        $('.about').addClass('hidden');
    });

})();

const structure = {
    'input': {
        defaults: {
            name: null,
            image: null,
            width: null,
            height: null,
            diameter: null,
            fov: null,
            offsetX: null,
            offsetY: null,
            scale: null
        },
        attr: {
            unique: true
        }
    },
    'output': {
        defaults: {
            width: null,
            height: null,
            image: null,
            imageIsValid: false
        },
        attr: {
            unique: true
        }
    },
    'state': {
        defaults: {
            activeRegion: -1
        },
        attr: {
            unique: true
        }
    },
    'constraint': {
        defaults: {
            minX: 0,
            minY: 0,
            maxX: 0,
            maxY: 0
        }
    },
    'regions': {
        defaults: {},
        attr: {
            autoIncrement: true
        }
    },
    'applied-regions': {
        defaults: {},
        attr: {
            autoIncrement: true
        }
    },
    'grids': {
        defaults: {},
        attr: {
            autoIncrement: true
        }
    },
    'settings': {
        defaults: {
            mode: 'planar420',
            eccMode: 'eccDisabled'
        }
    }
};
actions.deferAction('storage:remove-database', {databaseName: 'gdcDB'}).then(function() {
        return actions.deferAction('storage:create-database', {databaseName: 'gdcDB', structure: structure});
    }).then(function() {
        return actions.deferAction('settings:set-settings-data', {settingsData: {mode: 'planar420', eccMode: 'eccDisabled'}});
    }).then(function() {
        return actions.deferAction('output:set-output-resolution', {width: 1920, height: 1080});
    }).then(() => {
        events.dispatchEvent('regions:add-region');
    });

var confirmOnPageExit = function (e) {
    e = e || window.event;
    var message = 'Confirm';
    if (e) {
        e.returnValue = message;
    }
    return message;
};
window.onbeforeunload = confirmOnPageExit;

$('img.icon').each(function(){
    var $img = $(this);
    var imgID = $img.attr('id');
    var imgClass = $img.attr('class');
    var imgTitle = $img.attr('title');
    var imgClickEvent = $img.attr('data-click-event');
    var imgURL = $img.attr('src');

    $.get(imgURL, function(data) {
        var $svg = $(data).find('svg');
        if(typeof imgID !== 'undefined') {
            $svg = $svg.attr('id', imgID);
        }
        if(typeof imgClass !== 'undefined') {
            $svg = $svg.attr('class', imgClass+' replaced-svg');
        }
        if (imgClickEvent) {
            $svg.attr('data-click-event', imgClickEvent);
        }
        if (imgTitle) {
            $svg.attr('title', imgTitle);
        }
        $svg = $svg.removeAttr('xmlns:a');
        if(!$svg.attr('viewBox') && $svg.attr('height') && $svg.attr('width')) {
            $svg.attr('viewBox', '0 0 ' + $svg.attr('height') + ' ' + $svg.attr('width'));
        }
        $img.replaceWith($svg);
    }, 'xml');
});
